<?php

declare(strict_types=1);

namespace ContentReady\Util;

/**
 * 内容信号分析器：分析正文特征，供自适应规则判断使用。
 */
final class ContentAnalyzer
{
	/**
	 * 分析 HTML 内容并返回信号数组。
	 *
	 * @param string $html_content 正文 HTML
	 * @param string $title 文章标题
	 * @return array{
	 *     word_count: int,
	 *     has_structure: bool,
	 *     has_question: bool,
	 *     has_outbound_links: bool,
	 *     heading_count: int,
	 *     outbound_link_count: int
	 * }
	 */
	public static function analyze(string $html_content, string $title): array
	{
		// 去 HTML 标签后的纯文本
		$text = wp_strip_all_tags($html_content);
		$word_count = self::count_words($text);

		// 结构信号：H2/H3 数量
		$heading_count = self::count_headings($html_content);
		$has_structure = $heading_count >= 2;

		// 问句信号：标题或小标题包含问句特征
		$has_question = self::detect_question($html_content, $title);

		// 出站链接信号
		$outbound_links = self::count_outbound_links($html_content);
		$has_outbound_links = $outbound_links >= 1;

		return [
			'word_count'          => $word_count,
			'has_structure'       => $has_structure,
			'has_question'        => $has_question,
			'has_outbound_links'  => $has_outbound_links,
			'heading_count'       => $heading_count,
			'outbound_link_count' => $outbound_links,
		];
	}

	/**
	 * 计算字数（中文按字符计，英文按词计）。
	 */
	private static function count_words(string $text): int
	{
		$text = trim($text);
		if ($text === '') {
			return 0;
		}

		// 移除多余空白
		$text = preg_replace('/\s+/u', ' ', $text);
		if (!is_string($text)) {
			return 0;
		}

		// 中文字符数
		$cjk_count = 0;
		if (preg_match_all('/[\x{4e00}-\x{9fff}\x{3400}-\x{4dbf}]/u', $text, $matches)) {
			$cjk_count = count($matches[0]);
		}

		// 移除中文后统计英文词数
		$text_no_cjk = preg_replace('/[\x{4e00}-\x{9fff}\x{3400}-\x{4dbf}]/u', '', $text);
		if (!is_string($text_no_cjk)) {
			$text_no_cjk = '';
		}
		$words = preg_split('/\s+/', trim($text_no_cjk), -1, PREG_SPLIT_NO_EMPTY);
		$word_count = is_array($words) ? count($words) : 0;

		return $cjk_count + $word_count;
	}

	/**
	 * 统计 H2/H3 数量。
	 */
	private static function count_headings(string $html): int
	{
		$count = 0;
		if (preg_match_all('/<h[23][^>]*>/i', $html, $matches)) {
			$count = count($matches[0]);
		}
		return $count;
	}

	/**
	 * 检测问句信号：标题或小标题包含问号或疑问词。
	 */
	private static function detect_question(string $html, string $title): bool
	{
		// 问句特征正则
		$question_pattern = '/[?？]|怎么|如何|是否|为什么|什么|哪些|哪个|能否|可以吗/u';

		// 检查标题
		if (preg_match($question_pattern, $title)) {
			return true;
		}

		// 提取所有 H1-H6 标题文本并检查
		if (preg_match_all('/<h[1-6][^>]*>(.*?)<\/h[1-6]>/is', $html, $matches)) {
			foreach ($matches[1] as $heading_html) {
				$heading_text = wp_strip_all_tags((string)$heading_html);
				if (preg_match($question_pattern, $heading_text)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * 统计出站链接数量（排除站内链接）。
	 */
	private static function count_outbound_links(string $html): int
	{
		$count = 0;
		$site_host = (string)wp_parse_url(home_url(), PHP_URL_HOST);

		if (preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>/i', $html, $matches)) {
			foreach ($matches[1] as $url) {
				$url = (string)$url;
				// 跳过锚点链接
				if (strpos($url, '#') === 0) {
					continue;
				}
				// 跳过相对链接
				if (strpos($url, 'http') !== 0) {
					continue;
				}
				// 检查是否为外部链接
				$link_host = (string)wp_parse_url($url, PHP_URL_HOST);
				if ($link_host !== '' && $link_host !== $site_host) {
					$count++;
				}
			}
		}

		return $count;
	}
}
