<?php

declare(strict_types=1);

namespace ContentReady\Updates;

use ContentReady\Admin\Settings;

final class Updater
{
	private const SLUG = 'content-ready';
	private const TRANSIENT_KEY = 'cr_update_info_v1';

	private const DEFAULT_INFO_URL = 'https://cr-license.imsxx.com/v1/update/info';
	private const DEFAULT_LATEST_URL = 'https://cr-license.imsxx.com/v1/update/latest';

	public static function register(): void
	{
		add_filter('pre_set_site_transient_update_plugins', [self::class, 'inject_update'], 10, 1);
		add_filter('plugins_api', [self::class, 'plugins_api'], 10, 3);

		add_action('admin_post_cr_check_updates', [self::class, 'handle_manual_check']);
	}

	public static function info_url(): string
	{
		$url = (string)apply_filters('content_ready_update_info_url', self::DEFAULT_INFO_URL);
		$url = trim($url);
		return esc_url_raw($url);
	}

	public static function latest_url(): string
	{
		$url = (string)apply_filters('content_ready_update_latest_url', self::DEFAULT_LATEST_URL);
		$url = (string)apply_filters('content_ready_update_download_url', $url);
		$url = trim($url);
		return esc_url_raw($url);
	}

	/**
	 * @param mixed $transient
	 * @return mixed
	 */
	public static function inject_update($transient)
	{
		if (!is_object($transient) || !isset($transient->checked) || !is_array($transient->checked)) {
			return $transient;
		}

		$plugin = plugin_basename(CR_PLUGIN_FILE);
		$current = isset($transient->checked[$plugin]) ? (string)$transient->checked[$plugin] : (defined('CR_VERSION') ? (string)CR_VERSION : '');
		if ($current === '') {
			return $transient;
		}

		$info = self::get_update_info();
		if (empty($info['ok']) || empty($info['version'])) {
			return $transient;
		}

		$remote = sanitize_text_field((string)$info['version']);
		if ($remote === '' || version_compare($remote, $current, '<=')) {
			return $transient;
		}

		$item = new \stdClass();
		$item->slug = self::SLUG;
		$item->plugin = $plugin;
		$item->new_version = $remote;
		$item->url = sanitize_text_field((string)($info['details_url'] ?? ''));
		$package = self::latest_url();
		if ($package !== '') {
			$item->package = add_query_arg('v', $remote, $package);
		}

		$requires = sanitize_text_field((string)($info['requires_wp'] ?? ''));
		$requires_php = sanitize_text_field((string)($info['requires_php'] ?? ''));
		$tested = sanitize_text_field((string)($info['tested_wp'] ?? ''));
		if ($requires !== '') {
			$item->requires = $requires;
		}
		if ($requires_php !== '') {
			$item->requires_php = $requires_php;
		}
		if ($tested !== '') {
			$item->tested = $tested;
		}

		if (!isset($transient->response) || !is_array($transient->response)) {
			$transient->response = [];
		}
		$transient->response[$plugin] = $item;
		return $transient;
	}

	/**
	 * @param mixed  $result
	 * @param string $action
	 * @param mixed  $args
	 * @return mixed
	 */
	public static function plugins_api($result, string $action, $args)
	{
		if ($action !== 'plugin_information' || !is_object($args)) {
			return $result;
		}

		$slug = isset($args->slug) ? sanitize_key((string)$args->slug) : '';
		if ($slug !== self::SLUG) {
			return $result;
		}

		$info = self::get_update_info();
		$version = sanitize_text_field((string)($info['version'] ?? (defined('CR_VERSION') ? CR_VERSION : '')));
		$details_url = sanitize_text_field((string)($info['details_url'] ?? ''));
		$changelog = (string)($info['changelog'] ?? '');

		$obj = new \stdClass();
		$obj->name = 'Content Ready｜内容就绪';
		$obj->slug = self::SLUG;
		$obj->version = $version !== '' ? $version : (defined('CR_VERSION') ? (string)CR_VERSION : '');
		$obj->author = 'Content Ready';
		$obj->homepage = $details_url;

		$requires = sanitize_text_field((string)($info['requires_wp'] ?? ''));
		$requires_php = sanitize_text_field((string)($info['requires_php'] ?? ''));
		$tested = sanitize_text_field((string)($info['tested_wp'] ?? ''));
		if ($requires !== '') {
			$obj->requires = $requires;
		}
		if ($requires_php !== '') {
			$obj->requires_php = $requires_php;
		}
		if ($tested !== '') {
			$obj->tested = $tested;
		}

		$obj->sections = [
			'description' => '在编辑器内一键补齐“内容就绪（GEO）”模块，并输出一致的前端展示与结构化信息层。',
			'changelog' => $changelog !== '' ? wp_kses_post($changelog) : '暂无变更说明。',
		];

		$download = self::latest_url();
		if ($download !== '') {
			$obj->download_link = $download;
		}

		return $obj;
	}

	/**
	 * @return array{ok:bool,version:string,checked_at?:int,details_url?:string,changelog?:string,requires_wp?:string,requires_php?:string,tested_wp?:string}
	 */
	public static function get_update_info(bool $force = false): array
	{
		if ($force) {
			delete_site_transient(self::TRANSIENT_KEY);
		}

		$cached = get_site_transient(self::TRANSIENT_KEY);
		if (is_array($cached) && !empty($cached['ok']) && !empty($cached['version'])) {
			return $cached;
		}

		$url = self::info_url();
		if ($url === '') {
			return ['ok' => false, 'version' => ''];
		}

		$resp = wp_remote_get($url, [
			'timeout' => 8,
			'redirection' => 2,
			'headers' => [
				'Accept' => 'application/json',
				'User-Agent' => 'ContentReady/' . (defined('CR_VERSION') ? CR_VERSION : 'dev') . '; ' . home_url('/'),
			],
		]);

		if (is_wp_error($resp)) {
			return ['ok' => false, 'version' => ''];
		}

		$code = (int)wp_remote_retrieve_response_code($resp);
		$body = (string)wp_remote_retrieve_body($resp);
		$data = json_decode($body, true);
		$data = is_array($data) ? $data : [];

		if ($code < 200 || $code >= 300) {
			return ['ok' => false, 'version' => ''];
		}

		$version = sanitize_text_field((string)($data['version'] ?? ''));
		if ($version === '') {
			return ['ok' => false, 'version' => ''];
		}

		$out = [
			'ok' => true,
			'version' => $version,
			'checked_at' => time(),
		];

		foreach (['details_url', 'changelog', 'requires_wp', 'requires_php', 'tested_wp'] as $k) {
			if (isset($data[$k])) {
				$out[$k] = (string)$data[$k];
			}
		}

		set_site_transient(self::TRANSIENT_KEY, $out, 3600);
		return $out;
	}

	public static function get_cached_update_info(): array
	{
		$cached = get_site_transient(self::TRANSIENT_KEY);
		return is_array($cached) ? $cached : [];
	}

	public static function handle_manual_check(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have permission to access this page.', 'content-ready'));
		}

		check_admin_referer('cr_check_updates');

		$info = self::get_update_info(true);
		$ok = !empty($info['ok']);
		$remote = $ok ? sanitize_text_field((string)($info['version'] ?? '')) : '';
		$current = defined('CR_VERSION') ? (string)CR_VERSION : '';
		$status = 'failed';
		if ($ok && $remote !== '' && $current !== '') {
			$status = version_compare($remote, $current, '>') ? 'available' : 'latest';
		}

		// 同步刷新 WordPress 的插件更新 transient，确保插件列表能立即看到更新提示。
		if (!function_exists('wp_update_plugins')) {
			$path = ABSPATH . WPINC . '/update.php';
			if (file_exists($path)) {
				require_once $path;
			}
		}
		if (function_exists('delete_site_transient')) {
			delete_site_transient('update_plugins');
		}
		if (function_exists('wp_update_plugins')) {
			wp_update_plugins();
		}

		$args = [
			'page' => 'content-ready-settings',
			'tab' => 'license',
			'cr_notice' => 'cr_update_checked',
			'cr_update_status' => $status,
		];
		if ($remote !== '') {
			$args['cr_update_remote'] = $remote;
		}
		$checked_at = (int)($info['checked_at'] ?? 0);
		if ($checked_at > 0) {
			$args['cr_update_checked_at'] = $checked_at;
		}

		$url = add_query_arg($args, admin_url('admin.php'));
		wp_safe_redirect($url);
		exit;
	}
}
