<?php
declare(strict_types=1);

namespace ContentReady\Telemetry;

use ContentReady\Admin\Settings;

final class TelemetryRuntime
{
        private const META_OPTION = 'cr_telemetry_meta_v1';
        private const ACTIVATION_PING_HOOK = 'cr_telemetry_activation_ping';
        private const DAILY_PING_HOOK = 'cr_telemetry_daily_ping';

        public static function register(): void
        {
                add_action('init', [self::class, 'maybe_schedule_daily_ping'], 30);
                add_action(self::ACTIVATION_PING_HOOK, [self::class, 'run_activation_ping'], 10);
                add_action(self::DAILY_PING_HOOK, [self::class, 'run_daily_ping'], 10);
        }

        public static function on_activate(): void
        {
                self::seed_meta();
                self::schedule_activation_ping();
                self::maybe_schedule_daily_ping();
        }

        public static function on_deactivate(): void
        {
                wp_clear_scheduled_hook(self::DAILY_PING_HOOK);
                wp_clear_scheduled_hook(self::ACTIVATION_PING_HOOK);
        }

        private static function schedule_activation_ping(): void
        {
                if (!self::is_enabled()) {
                        wp_clear_scheduled_hook(self::ACTIVATION_PING_HOOK);
                        return;
                }

                self::seed_meta();

                // 激活埋点不是关键链路：改为排队 + 非阻塞，避免用户可感知卡顿。
                wp_clear_scheduled_hook(self::ACTIVATION_PING_HOOK);

                $delay = function_exists('wp_rand') ? (int)wp_rand(10, 180) : 20;
                $next = time() + max(10, $delay);
                wp_schedule_single_event($next, self::ACTIVATION_PING_HOOK);
        }

        public static function run_activation_ping(): void
        {
                self::send_ping('activate', true, false);
        }

        public static function maybe_schedule_daily_ping(): void
        {
                if (!self::is_enabled()) {
                        wp_clear_scheduled_hook(self::DAILY_PING_HOOK);
                        return;
                }

                self::seed_meta();

                $current = wp_next_scheduled(self::DAILY_PING_HOOK);
                if ($current) {
                        return;
                }

                $delay = function_exists('wp_rand') ? (int)wp_rand(10 * 60, 60 * 60) : (20 * 60);
                $next = time() + max(60, $delay);
                wp_schedule_event($next, 'daily', self::DAILY_PING_HOOK);
        }

        public static function run_daily_ping(): void
        {
                self::send_ping('daily_ping', false, false);
        }

        private static function send_ping(string $event, bool $force, bool $blocking): void
        {
                if (!self::is_enabled()) {
                        return;
                }

                $meta = self::get_meta();
                $install_id = isset($meta['install_id']) ? (string)$meta['install_id'] : '';
                $install_id = trim($install_id);
                if ($install_id === '') {
                        $install_id = self::seed_meta();
                }

                $now = time();
                $last_sent_at = isset($meta['last_sent_at']) ? (int)$meta['last_sent_at'] : 0;
                if (!$force && $last_sent_at > 0 && ($now - $last_sent_at) < DAY_IN_SECONDS) {
                        return;
                }

                $result = TelemetryClient::ping($install_id, $event, ['blocking' => $blocking]);

                $meta['last_sent_at'] = $now;
                $meta['last_http_status'] = (int)($result['http_status'] ?? 0);
                $meta['last_error'] = sanitize_text_field((string)($result['error'] ?? ''));
                update_option(self::META_OPTION, $meta, false);
        }

        private static function seed_meta(): string
        {
                $meta = self::get_meta();

                $install_id = isset($meta['install_id']) ? (string)$meta['install_id'] : '';
                $install_id = trim($install_id);
                if ($install_id === '') {
                        $install_id = function_exists('wp_generate_uuid4') ? (string)wp_generate_uuid4() : (string)uniqid('cr_', true);
                        $meta['install_id'] = $install_id;
                }

                if (!isset($meta['created_at']) || !is_int($meta['created_at']) || $meta['created_at'] <= 0) {
                        $meta['created_at'] = (int)current_time('timestamp');
                }

                update_option(self::META_OPTION, $meta, false);
                return $install_id;
        }

        private static function get_meta(): array
        {
                $meta = get_option(self::META_OPTION, []);
                return is_array($meta) ? $meta : [];
        }

        private static function is_enabled(): bool
        {
                $options = wp_parse_args(Settings::get_raw_options(), Settings::default_options());
                $privacy = is_array($options['privacy'] ?? null) ? $options['privacy'] : [];
                $enabled = !empty($privacy['telemetry_enabled']);

                $enabled = (bool)apply_filters('content_ready_telemetry_enabled', $enabled);
                return $enabled;
        }
}
