<?php
declare(strict_types=1);

namespace ContentReady\Telemetry;

final class TelemetryClient
{
        private const DEFAULT_PING_URL = 'https://cr-license.imsxx.com/v1/telemetry/ping';

        public static function ping_url(): string
        {
                $url = (string)apply_filters('content_ready_telemetry_ping_url', self::DEFAULT_PING_URL);
                $url = trim($url);
                return esc_url_raw($url);
        }

        /**
         * @return array{ok:bool,http_status:int,data:array<string,mixed>,error:string}
         */
        public static function ping(string $install_id, string $event, array $args = []): array
        {
                $url = self::ping_url();
                if ($url === '') {
                        return [
                                'ok' => false,
                                'http_status' => 0,
                                'data' => [],
                                'error' => 'missing_ping_url',
                        ];
                }

                $payload = [
                        'install_id' => $install_id,
                        'event' => $event,
                        'plugin_version' => defined('CR_VERSION') ? (string)CR_VERSION : '',
                        'wp_version' => get_bloginfo('version'),
                        'php_version' => PHP_VERSION,
                        'locale' => (string)get_locale(),
                ];

                $blocking = array_key_exists('blocking', $args) ? (bool)$args['blocking'] : true;
                $timeout = array_key_exists('timeout', $args) ? (float)$args['timeout'] : ($blocking ? 3 : 0.01);
                $timeout = max(0.01, min(10, $timeout));

                $resp = wp_remote_post($url, [
                        'timeout' => $timeout,
                        'blocking' => $blocking,
                        'headers' => [
                                'Content-Type' => 'application/json',
                        ],
                        'body' => wp_json_encode($payload),
                ]);

                if (is_wp_error($resp)) {
                        return [
                                'ok' => false,
                                'http_status' => 0,
                                'data' => [],
                                'error' => sanitize_text_field((string)$resp->get_error_message()),
                        ];
                }

                // 非阻塞：不等待响应/不解析内容，避免对用户请求链路产生可感知影响。
                if (!$blocking) {
                        return [
                                'ok' => true,
                                'http_status' => 0,
                                'data' => [],
                                'error' => '',
                        ];
                }

                $code = (int)wp_remote_retrieve_response_code($resp);
                $body = (string)wp_remote_retrieve_body($resp);
                $data = json_decode($body, true);
                $data = is_array($data) ? $data : [];

                if ($code < 200 || $code >= 300) {
                        return [
                                'ok' => false,
                                'http_status' => $code,
                                'data' => $data,
                                'error' => $data['error'] ?? ('http_' . $code),
                        ];
                }

                return [
                        'ok' => true,
                        'http_status' => $code,
                        'data' => $data,
                        'error' => '',
                ];
        }
}
