<?php

declare(strict_types=1);

namespace ContentReady\SiteEntrances;

final class WpSitemapDiagnostics
{
	private const TRANSIENT_KEY = 'cr_wp_sitemap_diag_v1';
	private const TRANSIENT_TTL = 10 * MINUTE_IN_SECONDS;

	/**
	 * @return array{checked_at:int,supported:bool,enabled:?bool,endpoint:string,http:array{ok:bool,status:int,content_type:string,looks_like_sitemap:?bool,error_code:string,error_message:string}}
	 */
	public static function get(bool $force_refresh = false): array
	{
		$cached = $force_refresh ? null : get_transient(self::TRANSIENT_KEY);
		if (is_array($cached)) {
			return self::normalize_snapshot($cached);
		}

		$snapshot = self::probe();
		set_transient(self::TRANSIENT_KEY, $snapshot, self::TRANSIENT_TTL);

		return $snapshot;
	}

	public static function clear_cache(): void
	{
		delete_transient(self::TRANSIENT_KEY);
	}

	/**
	 * @return array{checked_at:int,supported:bool,enabled:?bool,endpoint:string,http:array{ok:bool,status:int,content_type:string,looks_like_sitemap:?bool,error_code:string,error_message:string}}
	 */
	private static function probe(): array
	{
		$endpoint = (string)home_url('/wp-sitemap.xml');

		$supported = function_exists('wp_sitemaps_enabled') && function_exists('wp_sitemaps_get_server');
		$enabled = null;
		if (function_exists('wp_sitemaps_enabled')) {
			$enabled = (bool)wp_sitemaps_enabled();
		}

		$http = [
			'ok' => false,
			'status' => 0,
			'content_type' => '',
			'looks_like_sitemap' => null,
			'error_code' => '',
			'error_message' => '',
		];

		if (!function_exists('wp_remote_get')) {
			$http['error_code'] = 'wp_remote_get_missing';
			$http['error_message'] = '站点未启用 WP HTTP API，无法进行回环检测。';
			return [
				'checked_at' => time(),
				'supported' => $supported,
				'enabled' => $enabled,
				'endpoint' => $endpoint,
				'http' => $http,
			];
		}

		$res = wp_remote_get(
			$endpoint,
			[
				'timeout' => 3,
				'redirection' => 1,
				'headers' => [
					'Accept' => 'application/xml,text/xml,*/*;q=0.8',
				],
				'user-agent' => 'ContentReady/' . (defined('CR_VERSION') ? (string)CR_VERSION : 'dev'),
			]
		);

		if (is_wp_error($res)) {
			$http['error_code'] = sanitize_key((string)$res->get_error_code());
			$http['error_message'] = sanitize_text_field((string)$res->get_error_message());
			return [
				'checked_at' => time(),
				'supported' => $supported,
				'enabled' => $enabled,
				'endpoint' => $endpoint,
				'http' => $http,
			];
		}

		$status = (int)wp_remote_retrieve_response_code($res);
		$content_type = (string)wp_remote_retrieve_header($res, 'content-type');
		$body = (string)wp_remote_retrieve_body($res);

		$looks_like = null;
		if ($body !== '') {
			$head = substr($body, 0, 4096);
			$looks_like = stripos($head, '<sitemapindex') !== false || stripos($head, '<urlset') !== false;
		}

		$http['ok'] = true;
		$http['status'] = $status;
		$http['content_type'] = sanitize_text_field($content_type);
		$http['looks_like_sitemap'] = $looks_like;

		return [
			'checked_at' => time(),
			'supported' => $supported,
			'enabled' => $enabled,
			'endpoint' => $endpoint,
			'http' => $http,
		];
	}

	private static function normalize_snapshot(array $raw): array
	{
		$checked_at = (int)($raw['checked_at'] ?? 0);
		$supported = !empty($raw['supported']);
		$enabled = array_key_exists('enabled', $raw) ? (is_bool($raw['enabled']) ? $raw['enabled'] : null) : null;
		$endpoint = sanitize_text_field((string)($raw['endpoint'] ?? ''));

		$http_raw = is_array($raw['http'] ?? null) ? $raw['http'] : [];
		$http = [
			'ok' => !empty($http_raw['ok']),
			'status' => (int)($http_raw['status'] ?? 0),
			'content_type' => sanitize_text_field((string)($http_raw['content_type'] ?? '')),
			'looks_like_sitemap' => array_key_exists('looks_like_sitemap', $http_raw) && is_bool($http_raw['looks_like_sitemap']) ? $http_raw['looks_like_sitemap'] : null,
			'error_code' => sanitize_key((string)($http_raw['error_code'] ?? '')),
			'error_message' => sanitize_text_field((string)($http_raw['error_message'] ?? '')),
		];

		return [
			'checked_at' => $checked_at,
			'supported' => $supported,
			'enabled' => $enabled,
			'endpoint' => $endpoint,
			'http' => $http,
		];
	}
}

