<?php

declare(strict_types=1);

namespace ContentReady\SiteEntrances;

use ContentReady\Admin\Settings;

/**
 * “最小实践”：基于 WordPress 自带 XML Sitemap（wp-sitemap.xml）做开关控制。
 *
 * 说明：
 * - 不实现自定义 sitemap（避免造概念、避免与 SEO 插件冲突）。
 * - 仅控制 WP 默认四项：文章/页面/标签/用户。
 * - 更细腻的 sitemap 管理建议交给专业 SEO/Sitemap 插件。
 */
final class WpSitemapController
{
	public static function register(): void
	{
		add_filter('wp_sitemaps_add_provider', [self::class, 'filter_add_provider'], 20, 2);
		add_filter('wp_sitemaps_post_types', [self::class, 'filter_post_types'], 20);
		add_filter('wp_sitemaps_taxonomies', [self::class, 'filter_taxonomies'], 20);
	}

	/**
	 * @param mixed $provider
	 * @return mixed
	 */
	public static function filter_add_provider($provider, string $name)
	{
		$se = self::site_entrances_options();
		if (empty($se['wp_sitemap_control_enabled'])) {
			return $provider;
		}

		$name = sanitize_key($name);
		if ($name !== 'users') {
			return $provider;
		}

		if (!empty($se['wp_sitemap_include_users'])) {
			return $provider;
		}

		return false;
	}

	/**
	 * @param array<string, \WP_Post_Type> $post_types
	 * @return array<string, \WP_Post_Type>
	 */
	public static function filter_post_types(array $post_types): array
	{
		$se = self::site_entrances_options();
		if (empty($se['wp_sitemap_control_enabled'])) {
			return $post_types;
		}

		// 仅控制 WordPress 默认的 post/page，不影响自定义 post type。
		if (empty($se['wp_sitemap_include_posts']) && isset($post_types['post'])) {
			unset($post_types['post']);
		}
		if (empty($se['wp_sitemap_include_pages']) && isset($post_types['page'])) {
			unset($post_types['page']);
		}

		return $post_types;
	}

	/**
	 * @param array<string, \WP_Taxonomy> $taxonomies
	 * @return array<string, \WP_Taxonomy>
	 */
	public static function filter_taxonomies(array $taxonomies): array
	{
		$se = self::site_entrances_options();
		if (empty($se['wp_sitemap_control_enabled'])) {
			return $taxonomies;
		}

		// 仅控制 WordPress 默认的 post_tag，不影响其它 taxonomy。
		if (empty($se['wp_sitemap_include_tags']) && isset($taxonomies['post_tag'])) {
			unset($taxonomies['post_tag']);
		}

		return $taxonomies;
	}

	private static function site_entrances_options(): array
	{
		$raw = Settings::get_raw_options();
		$se_raw = is_array($raw['site_entrances'] ?? null) ? $raw['site_entrances'] : [];

		$defaults = Settings::default_options();
		$se_defaults = is_array($defaults['site_entrances'] ?? null) ? $defaults['site_entrances'] : [];

		$se = wp_parse_args($se_raw, $se_defaults);

		// 兼容旧版本：曾经存在自定义 ai-sitemap 开关，这里尽量映射到“接管 WP sitemap”。
		if (!array_key_exists('wp_sitemap_control_enabled', $se_raw) && array_key_exists('ai_sitemap_enabled', $se_raw)) {
			$se['wp_sitemap_control_enabled'] = !empty($se_raw['ai_sitemap_enabled']);
		}
		foreach (
			[
				'posts' => 'wp_sitemap_include_posts',
				'pages' => 'wp_sitemap_include_pages',
				'tags' => 'wp_sitemap_include_tags',
				'users' => 'wp_sitemap_include_users',
			] as $k => $new_key
		) {
			$old_key = 'ai_sitemap_include_' . $k;
			if (!array_key_exists($new_key, $se_raw) && array_key_exists($old_key, $se_raw)) {
				$se[$new_key] = !empty($se_raw[$old_key]);
			}
		}

		return is_array($se) ? $se : [];
	}
}

