<?php

declare(strict_types=1);

namespace ContentReady\SiteEntrances;

use ContentReady\Admin\Settings;

final class LlmsTxt
{
	private const MAX_POST_ITEMS = 20;
	private const MAX_PAGE_ITEMS = 20;

	public static function register(): void
	{
		add_action('template_redirect', [self::class, 'maybe_render'], 0);
	}

	public static function maybe_render(): void
	{
		if (!self::is_llms_txt_request()) {
			return;
		}

		$options = Settings::get_options();
		$se = is_array($options['site_entrances'] ?? null) ? $options['site_entrances'] : [];
		if (empty($se['llms_enabled'])) {
			return;
		}

		$md = self::build_markdown($options);

		nocache_headers();
		status_header(200);
		header('Content-Type: text/markdown; charset=utf-8');

		if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) === 'HEAD') {
			exit;
		}

		echo $md;
		exit;
	}

	private static function is_llms_txt_request(): bool
	{
		$req = (string)($_SERVER['REQUEST_URI'] ?? '');
		$req_path = (string)(wp_parse_url($req, PHP_URL_PATH) ?? '');
		$want_path = (string)(wp_parse_url(home_url('/llms.txt'), PHP_URL_PATH) ?? '');

		$req_path = rtrim($req_path, '/');
		$want_path = rtrim($want_path, '/');

		return $req_path !== '' && $want_path !== '' && strtolower($req_path) === strtolower($want_path);
	}

	private static function build_markdown(array $options): string
	{
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];

		$title = trim((string)($site['display_name'] ?? ''));
		if ($title === '') {
			$title = (string)get_bloginfo('name');
		}
		if ($title === '') {
			$title = 'Website';
		}

		$desc = trim((string)get_bloginfo('description'));
		if ($desc === '') {
			$desc = '站点内容清单与推荐页面入口（由 Content Ready 自动生成）。';
		}

		$se = is_array($options['site_entrances'] ?? null) ? $options['site_entrances'] : [];
		$recommended_raw = (string)($se['recommended_urls'] ?? '');
		$extra = UrlList::parse_recommended($recommended_raw);

		$urls = self::default_pages();
		foreach ($extra as $row) {
			if (!is_array($row)) {
				continue;
			}
			$url = trim((string)($row['url'] ?? ''));
			$label = trim((string)($row['label'] ?? ''));
			if ($url === '' || $label === '') {
				continue;
			}
			$urls[$url] = $label;
		}

		$lines = [];
		$lines[] = 'Generated by Content Ready, this is an llms.txt file designed to help LLMs better understand and index this website.';
		$lines[] = '';
		$lines[] = '# ' . $title;
		$lines[] = '';
		$lines[] = '> ' . str_replace(["\r", "\n"], ' ', $desc);
		$lines[] = '';

		$wp_sitemap_url = home_url('/wp-sitemap.xml');
		$lines[] = '## Sitemaps';
		$lines[] = '[' . $wp_sitemap_url . '](' . $wp_sitemap_url . '): WordPress XML Sitemap（如使用 SEO 插件可能会接管/替换）';
		$lines[] = '';

		$lines[] = '## 推荐页面（关键入口）';
		foreach ($urls as $url => $label) {
			$lines[] = '- [' . $label . '](' . $url . ')';
		}
		$lines[] = '';

		$posts = self::latest_posts('post', self::MAX_POST_ITEMS);
		if ($posts !== []) {
			$lines[] = '## 文章';
			foreach ($posts as $row) {
				$u = trim((string)($row['url'] ?? ''));
				$t = trim((string)($row['title'] ?? ''));
				$s = trim((string)($row['summary'] ?? ''));
				if ($u === '' || $t === '') {
					continue;
				}
				$line = '- [' . $t . '](' . $u . ')';
				if ($s !== '') {
					$line .= ': ' . $s;
				}
				$lines[] = $line;
			}
			$lines[] = '';
		}

		$pages = self::latest_posts('page', self::MAX_PAGE_ITEMS);
		if ($pages !== []) {
			$lines[] = '## 页面';
			foreach ($pages as $row) {
				$u = trim((string)($row['url'] ?? ''));
				$t = trim((string)($row['title'] ?? ''));
				$s = trim((string)($row['summary'] ?? ''));
				if ($u === '' || $t === '') {
					continue;
				}
				$line = '- [' . $t . '](' . $u . ')';
				if ($s !== '') {
					$line .= ': ' . $s;
				}
				$lines[] = $line;
			}
			$lines[] = '';
		}

		return implode("\n", $lines);
	}

	/**
	 * @return array<int, array{url:string,title:string,summary:string}>
	 */
	private static function latest_posts(string $post_type, int $limit): array
	{
		$post_type = sanitize_key($post_type);
		$limit = max(0, min(200, (int)$limit));
		if ($post_type === '' || $limit <= 0) {
			return [];
		}

		$q = new \WP_Query([
			'post_type' => $post_type,
			'post_status' => 'publish',
			'posts_per_page' => $limit,
			'orderby' => 'modified',
			'order' => 'DESC',
			'fields' => 'ids',
			'no_found_rows' => true,
			'ignore_sticky_posts' => true,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
		]);

		$ids = is_array($q->posts ?? null) ? $q->posts : [];
		$out = [];
		foreach ($ids as $id) {
			$id = (int)$id;
			if ($id <= 0) {
				continue;
			}

			$url = get_permalink($id);
			if (!is_string($url) || $url === '') {
				continue;
			}

			$title = (string)get_the_title($id);
			$title = trim(wp_strip_all_tags($title));
			if ($title === '') {
				continue;
			}

			$excerpt = trim(wp_strip_all_tags((string)get_post_field('post_excerpt', $id)));
			$content = trim(wp_strip_all_tags((string)get_post_field('post_content', $id)));
			$base = $excerpt !== '' ? $excerpt : $content;

			$summary = '';
			if ($base !== '') {
				$summary = wp_html_excerpt($base, 160, '…');
				$summary = preg_replace('/\\s+/', ' ', (string)$summary);
				$summary = trim((string)$summary);
			}

			$out[] = [
				'url' => (string)esc_url_raw($url),
				'title' => $title,
				'summary' => $summary,
			];
		}

		return $out;
	}

	/**
	 * @return array<string, string> url => label
	 */
	private static function default_pages(): array
	{
		$out = [];

		$home = home_url('/');
		if ($home !== '') {
			$out[$home] = '首页';
		}

		$show_on_front = (string)get_option('show_on_front', 'posts');
		if ($show_on_front === 'page') {
			$page_for_posts = (int)get_option('page_for_posts', 0);
			if ($page_for_posts > 0) {
				$url = get_permalink($page_for_posts);
				if (is_string($url) && $url !== '') {
					$out[$url] = '文章列表';
				}
			}
		}

		return $out;
	}
}
