<?php

declare(strict_types=1);

namespace ContentReady\Rest;

use ContentReady\Rest\Post\Checks;
use ContentReady\Rest\Post\Meta;
use ContentReady\Rest\Post\Modules;
use ContentReady\Rest\Post\Slug;

final class PostController
{
	public static function register(): void
	{
		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/meta', [
			[
				'methods' => 'GET',
				'callback' => [Meta::class, 'get_meta'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/page-type', [
			[
				'methods' => 'POST',
				'callback' => [Meta::class, 'set_page_type'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'page_type' => ['type' => 'string', 'required' => true],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/author', [
			[
				'methods' => 'POST',
				'callback' => [Meta::class, 'set_author'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'mode' => ['type' => 'string', 'required' => true],
					'name' => ['type' => 'string', 'required' => false],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/update-module/(?P<module_id>[a-z_]+)', [
			[
				'methods' => 'POST',
				'callback' => [Modules::class, 'update_module'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'data' => ['type' => 'object', 'required' => true],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/generate', [
			[
				'methods' => 'POST',
				'callback' => [Modules::class, 'generate_all'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'only_missing' => ['type' => 'boolean', 'required' => false],
					'skip_edited' => ['type' => 'boolean', 'required' => false],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/generate-module/(?P<module_id>[a-z_]+)', [
			[
				'methods' => 'POST',
				'callback' => [Modules::class, 'generate_module'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/check', [
			[
				'methods' => 'POST',
				'callback' => [Checks::class, 'check_post'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/structured-preview', [
			[
				'methods' => 'POST',
				'callback' => [Checks::class, 'structured_preview'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/apply-slug', [
			[
				'methods' => 'POST',
				'callback' => [Slug::class, 'apply_slug'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'slug' => ['type' => 'string', 'required' => true],
					'ack_published' => ['type' => 'boolean', 'required' => false],
				],
			],
		]);
	}

	public static function can_edit_post(\WP_REST_Request $request): bool
	{
		$post_id = (int)$request['id'];
		return $post_id > 0 && current_user_can('edit_post', $post_id);
	}
}
