<?php

declare(strict_types=1);

namespace ContentReady\Rest\Post;

use ContentReady\Admin\Settings;
use ContentReady\Licensing\Capabilities;

final class Slug
{
	public static function apply_slug(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_APPLY_SLUG, $options)) {
			return new \WP_Error('cr_pro_required', '该功能仅 Pro 可用：一键应用 slug。', ['status' => 403]);
		}

		$post = get_post($post_id);
		if (!$post) {
			return new \WP_Error('cr_post_not_found', '文章不存在或已删除。', ['status' => 404]);
		}

		$raw = (string)$request->get_param('slug');
		$requested = self::sanitize_slug_input($raw);
		if ($requested === '') {
			return new \WP_Error('cr_invalid_slug', 'Slug 无效：请使用英文单词与连字符（a-z0-9-）。', ['status' => 400]);
		}

		$current = (string)$post->post_name;
		if ($requested === $current) {
			return rest_ensure_response([
				'post_id' => $post_id,
				'requested' => $requested,
				'applied' => $current,
				'permalink' => (string)get_permalink($post),
			]);
		}

		$is_published = ((string)$post->post_status) === 'publish';
		$ack = (bool)$request->get_param('ack_published');
		if ($is_published && !$ack) {
			return new \WP_Error(
				'cr_slug_requires_ack',
				'本文已经发布，修改 slug 可能影响 SEO。请确认后再应用。',
				['status' => 400]
			);
		}

		$unique = wp_unique_post_slug($requested, $post_id, (string)$post->post_status, (string)$post->post_type, (int)$post->post_parent);
		$updated = wp_update_post([
			'ID' => $post_id,
			'post_name' => $unique,
		], true);

		if (is_wp_error($updated)) {
			return $updated;
		}

		$updated_post = get_post($post_id);
		return rest_ensure_response([
			'post_id' => $post_id,
			'requested' => $requested,
			'applied' => $updated_post ? (string)$updated_post->post_name : $unique,
			'permalink' => $updated_post ? (string)get_permalink($updated_post) : (string)get_permalink($post_id),
		]);
	}

	private static function sanitize_slug_input(string $slug): string
	{
		$slug = sanitize_title($slug);
		$slug = strtolower($slug);
		$slug = preg_replace('/[^a-z0-9-]+/', '-', $slug) ?? '';
		$slug = preg_replace('/-+/', '-', $slug) ?? '';
		$slug = trim($slug, '-');

		if (strlen($slug) > 80) {
			$slug = substr($slug, 0, 80);
			$slug = rtrim($slug, '-');
		}

		return $slug;
	}
}

