<?php

declare(strict_types=1);

namespace ContentReady\Rest\Post\Modules;

use ContentReady\Admin\Settings;
use ContentReady\Generator\V1Generator;
use ContentReady\Licensing\Capabilities;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\Modules\Sanitizer;
use ContentReady\Rest\Post\Meta;

final class GenerateModule
{
	public static function handle(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$module_id = sanitize_key((string)$request['module_id']);
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return new \WP_Error('cr_invalid_module', '模块无效。', ['status' => 400]);
		}

		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_SINGLE_MODULE_GENERATE, $options)) {
			return new \WP_Error('cr_pro_required', '该功能仅 Pro 可用：单模块生成。', ['status' => 403]);
		}

		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
		if (empty($cfg['enabled'])) {
			return new \WP_Error('cr_module_disabled', '该模块未启用。', ['status' => 400]);
		}
		$generator = new V1Generator($options);
		$meta = MetaStore::get($post_id);
		$target_ids = $generator->get_target_module_ids($meta);
		if (!in_array($module_id, $target_ids, true)) {
			return new \WP_Error('cr_module_not_in_page_type', '该模块不属于当前页面类型的模块组合。', ['status' => 400]);
		}
		if ($generator->requires_ai($module_id) && !$generator->is_ai_configured()) {
			return new \WP_Error('cr_ai_not_configured', 'AI 未配置：请到设置页填写 AI 接口地址与默认模型。', ['status' => 400]);
		}

		$data = $generator->generate_module($post_id, $module_id, $meta);
		if ($data === []) {
			$detail = $generator->get_ai_last_error();
			return new \WP_Error('cr_generate_failed', '生成失败：' . sanitize_text_field($detail !== '' ? $detail : 'empty_result'), ['status' => 500]);
		}
		$clean = Sanitizer::sanitize_module_data($module_id, $data);
		MetaStore::touch_generated($post_id, $module_id, $clean);
		SummaryExcerptWriter::maybe_write($post_id, $modules_cfg, $module_id, $clean);

		return Meta::get_meta($request);
	}
}

