<?php

declare(strict_types=1);

namespace ContentReady\Rest\Post\Modules;

use ContentReady\Admin\Settings;
use ContentReady\Generator\V1Generator;
use ContentReady\Licensing\Capabilities;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\Modules\Sanitizer;
use ContentReady\Rest\Post\Meta;

final class GenerateAll
{
	public static function handle(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$only_missing = (bool)$request->get_param('only_missing');
		$skip_edited_param = $request->get_param('skip_edited');
		$skip_edited = $skip_edited_param === null ? true : (bool)$skip_edited_param;
		$options = Settings::get_options();
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$generator = new V1Generator($options);

		$meta = MetaStore::get($post_id);
		$target_module_ids = $generator->get_target_module_ids($meta);
		$eligible_modules = [];

		$needs_ai = false;
		foreach ($target_module_ids as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			if (!Capabilities::is_module_available($module_id, $options)) {
				continue;
			}
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}

			$eligible_modules[$module_id] = $cfg;
			if ($generator->requires_ai($module_id)) {
				$needs_ai = true;
			}
		}

		if ($needs_ai && !$generator->is_ai_configured()) {
			return new \WP_Error('cr_ai_not_configured', 'AI 未配置：请到设置页填写 AI 接口地址与默认模型。', ['status' => 400]);
		}

		// 收集生成结果
		$generated = [];
		$skipped = [];
		$failed = [];
		$generated_payloads = [];

		foreach ($eligible_modules as $module_id => $cfg) {
			if ($only_missing && !self::is_missing($meta['modules'][$module_id]['data'] ?? null)) {
				continue;
			}
			if ($skip_edited) {
				$m = is_array($meta['modules'][$module_id] ?? null) ? $meta['modules'][$module_id] : [];
				$gen_at = (int)($m['generated_at'] ?? 0);
				$edit_at = (int)($m['edited_at'] ?? 0);
				if ($edit_at > 0 && $gen_at > 0 && $edit_at > $gen_at) {
					$label = (string)($cfg['label'] ?? $module_id);
					$skipped[] = ['id' => $module_id, 'label' => $label, 'reason' => '已手动编辑'];
					continue;
				}
			}
			$data = $generator->generate_module($post_id, $module_id, $meta);

			// 处理跳过的模块
			if (!empty($data['_skipped'])) {
				$label = (string)($cfg['label'] ?? $module_id);
				$reason = (string)($data['_reason'] ?? '自适应规则');
				$skipped[] = ['id' => $module_id, 'label' => $label, 'reason' => $reason];
				continue;
			}

			if ($data === []) {
				$detail = $generator->get_ai_last_error();
				$failed[] = ['id' => $module_id, 'error' => $detail];
				continue;
			}

			$clean = Sanitizer::sanitize_module_data($module_id, $data);
			$generated_payloads[$module_id] = $clean;
			SummaryExcerptWriter::maybe_write($post_id, $modules_cfg, $module_id, $clean);
			$generated[] = $module_id;
		}
		if ($generated_payloads !== []) {
			$meta = MetaStore::touch_generated_batch($post_id, $generated_payloads);
		}

		// 构建响应消息
		$response = Meta::get_meta($request);
		if (is_wp_error($response)) {
			return $response;
		}

		$response_data = $response->get_data();
		$response_data['generation_result'] = [
			'generated' => $generated,
			'skipped'   => $skipped,
			'failed'    => $failed,
		];

		$response_data['generation_message'] = self::build_generation_message($generated, $skipped, $failed);
		self::maybe_override_all_skipped_message($response_data, $generated, $skipped, $failed);

		return rest_ensure_response($response_data);
	}

	private static function is_missing($data): bool
	{
		return !is_array($data) || $data === [];
	}

	private static function build_generation_message(array $generated, array $skipped, array $failed): string
	{
		$message_parts = [];
		if (count($generated) > 0) {
			$message_parts[] = sprintf('已生成 %d 个模块', count($generated));
		}
		if (count($skipped) > 0) {
			$skip_labels = array_map(fn($s) => $s['label'], $skipped);
			$message_parts[] = sprintf('跳过 %d 个模块（%s）', count($skipped), implode('、', $skip_labels));
		}
		if (count($failed) > 0) {
			$message_parts[] = sprintf('%d 个模块生成失败', count($failed));
		}

		return empty($message_parts) ? '没有需要生成的模块。' : (implode('；', $message_parts) . '。');
	}

	private static function maybe_override_all_skipped_message(array &$response_data, array $generated, array $skipped, array $failed): void
	{
		if (count($skipped) === 0 || count($generated) !== 0 || count($failed) !== 0) {
			return;
		}

		$all_edited = true;
		foreach ($skipped as $s) {
			$reason = is_array($s) ? (string)($s['reason'] ?? '') : '';
			if ($reason !== '已手动编辑') {
				$all_edited = false;
				break;
			}
		}

		if ($all_edited) {
			$response_data['generation_message'] = '没有需要生成的模块：已跳过所有手动编辑模块。';
			return;
		}

		$response_data['generation_message'] = '正文较短，已跳过部分模块的生成。摘要、锚点等必要模块不受影响。如需强制生成所有已启用模块，请前往设置将「模块调度策略」切换为“关闭”。';
	}
}
