<?php

declare(strict_types=1);

namespace ContentReady\Rest\Post;

use ContentReady\Admin\Settings;
use ContentReady\Licensing\Capabilities;
use ContentReady\Meta\MetaStore;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class Meta
{
	public static function get_meta(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$post = get_post($post_id);
		if (!$post) {
			return new \WP_Error('cr_post_not_found', '文章不存在或已删除。', ['status' => 404]);
		}
		$meta = MetaStore::get($post_id);
		$options = Settings::get_options();

		return rest_ensure_response([
			'post_id' => $post_id,
			'post' => [
				'slug' => (string)$post->post_name,
				'status' => (string)$post->post_status,
				'permalink' => (string)get_permalink($post),
			],
			'meta' => $meta,
			'options' => [
				'site' => $options['site'] ?? [],
				'modules_order' => $options['modules_order'] ?? [],
				'modules' => $options['modules'] ?? [],
				'page_types' => $options['page_types'] ?? [],
				'structured_output' => $options['structured_output'] ?? [],
				'capabilities' => Capabilities::client_caps($options),
			],
		]);
	}

	public static function set_page_type(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$page_type = sanitize_key((string)$request->get_param('page_type'));
		if ($page_type === '') {
			return new \WP_Error('cr_invalid_page_type', '页面类型无效。', ['status' => 400]);
		}

		$options = Settings::get_options();
		if (!PageTypesConfig::is_known_page_type($page_type, $options)) {
			return new \WP_Error('cr_unknown_page_type', '页面类型不存在，请先在设置页配置。', ['status' => 400]);
		}

		$meta = MetaStore::get($post_id);
		$meta['page_type'] = $page_type;
		MetaStore::update($post_id, $meta);

		return self::get_meta($request);
	}

	public static function set_author(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_AUTHOR_STRATEGY, $options)) {
			return new \WP_Error('cr_pro_required', '该功能仅 Pro 可用：作者策略。', ['status' => 403]);
		}

		$mode = sanitize_key((string)$request->get_param('mode'));
		if (!in_array($mode, ['inherit', 'wp', 'brand'], true)) {
			return new \WP_Error('cr_invalid_author_mode', '作者策略无效。', ['status' => 400]);
		}

		$name = sanitize_text_field((string)$request->get_param('name'));

		$meta = MetaStore::get($post_id);
		$meta['author'] = [
			'mode' => $mode,
			'name' => $name,
		];
		MetaStore::update($post_id, $meta);

		return self::get_meta($request);
	}
}

