<?php

declare(strict_types=1);

namespace ContentReady\Rest\Assessment;

use ContentReady\Admin\Settings;
use ContentReady\Frontend\StructuredData;
use ContentReady\Meta\MetaStore;

final class PostAssessment
{
	public static function assess(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$post = get_post($post_id);
		if (!$post) {
			return new \WP_Error('cr_post_not_found', 'post_not_found', ['status' => 404]);
		}

		if ((string)$post->post_status !== 'publish') {
			return new \WP_Error('cr_post_not_publish', 'post_not_publish', ['status' => 400]);
		}

		$permalink = get_permalink($post_id);
		if (!is_string($permalink) || $permalink === '') {
			return new \WP_Error('cr_permalink_missing', 'permalink_missing', ['status' => 400]);
		}

		$options = Settings::get_options();
		$preview = StructuredData::preview($post_id);
		$blocked_reason = sanitize_key((string)($preview['blocked_reason'] ?? ''));
		$meta = MetaStore::get($post_id);
		$expected_enabled_modules_with_data = ModuleInspector::collect_enabled_modules_with_data($meta, $options);
		$missing_enabled_modules = ModuleInspector::collect_enabled_modules_missing_data($meta, $options);
		$missing_enabled_modules_detail = ModuleInspector::module_details($missing_enabled_modules);
		$geo_modules = ModuleInspector::analyze_geo_modules($meta);

		$out = [
			'post_id' => $post_id,
			'permalink' => $permalink,
			'options' => [
				'inject_enabled' => !empty($options['frontend']['inject_enabled']),
				'structured_enabled' => !empty($options['structured_output']['enabled']),
			],
			'structured_preview' => [
				'would_output' => !empty($preview['would_output']),
				'blocked_reason' => $blocked_reason,
			],
			'jsonld' => [
				'total_scripts' => 0,
				'cr_scripts' => 0,
				'other_scripts' => 0,
				'invalid_json' => 0,
				'types' => [],
			],
			'content' => [
				'found_cr_modules' => false,
				'cr_module_count' => 0,
				'expected_enabled_modules_with_data' => $expected_enabled_modules_with_data,
				'expected_enabled_modules_count' => count($expected_enabled_modules_with_data),
				'missing_enabled_modules' => $missing_enabled_modules,
				'missing_enabled_modules_count' => count($missing_enabled_modules),
				'missing_enabled_modules_detail' => $missing_enabled_modules_detail,
			],
			'geo_modules' => $geo_modules,
			'warnings' => [],
			'suggestions' => [],
		];

		$response = wp_remote_get($permalink, [
			'timeout' => 12,
			'redirection' => 3,
			'headers' => [
				'User-Agent' => 'ContentReady/' . (defined('CR_VERSION') ? CR_VERSION : 'dev') . '; ' . home_url('/'),
			],
		]);

		if (is_wp_error($response)) {
			$out['warnings'][] = '抓取前端页面失败（常见原因：站点 loopback 被禁用 / 被安全插件或 WAF 拦截 / 站点需要登录）：' . sanitize_text_field($response->get_error_message()) . '。建议：先到「工具 → 站点健康」检查 loopback；必要时临时关闭安全插件/防火墙规则后重试。';
			$out['suggestions'] = Suggestions::default_suggestions($post_id, $blocked_reason);
			return $out;
		}

		$code = (int)wp_remote_retrieve_response_code($response);
		$body = (string)wp_remote_retrieve_body($response);
		if ($code < 200 || $code >= 300) {
			$out['warnings'][] = '前端页面返回非 2xx：HTTP ' . (string)$code . '（可能被鉴权/防火墙/维护模式拦截，导致检查结果不可信）。';
		}
		if (trim($body) === '') {
			$out['warnings'][] = '前端页面返回空内容（常见原因：安全插件/WAF 拦截，或站点禁用了 loopback 请求）。建议：到「工具 → 站点健康」查看 loopback；或临时停用安全插件/开启白名单后重试。';
			$out['suggestions'] = Suggestions::default_suggestions($post_id, $blocked_reason);
			return $out;
		}

		$jsonld = JsonLdAnalyzer::analyze($body);
		$out['jsonld'] = $jsonld;

		$content = ContentInjectionAnalyzer::analyze_with_base($body, $out['content']);
		$out['content'] = $content;

		$inject_enabled = !empty($options['frontend']['inject_enabled']);
		$expected_count = (int)($content['expected_enabled_modules_count'] ?? 0);
		if ($inject_enabled && empty($content['found_cr_modules'])) {
			if ($expected_count <= 0) {
				$out['warnings'][] = '前端未检测到 Content Ready 模块标记：该文章目前没有任何“已生成且启用”的模块（可在编辑器点击「生成 GEO」后再检查）。';
			} else {
				$out['warnings'][] = '设置中已开启前端渲染（基于 the_content），且该文章存在可展示模块，但前端未检测到 Content Ready 模块标记：主题可能未渲染 the_content，或被其他插件覆盖。';
			}
		}

		if (!empty($options['structured_output']['enabled']) && empty($preview['would_output']) && $blocked_reason !== '') {
			$out['warnings'][] = Suggestions::format_structured_block_warning($blocked_reason);
		}

		if (($jsonld['other_scripts'] ?? 0) > 0 && ($jsonld['cr_scripts'] ?? 0) > 0) {
			$out['warnings'][] = '检测到“本插件 + 其他来源”同时输出结构化（JSON-LD）：如有重复（同类 schema），建议在 SEO 插件中关闭结构化输出（Schema/JSON-LD），或升级 Pro 使用“强制输出”。';
		}

		$out['suggestions'] = Suggestions::default_suggestions($post_id, $blocked_reason);
		return $out;
	}
}

