<?php

declare(strict_types=1);

namespace ContentReady\Rest\Assessment;

use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Registry;

final class ModuleInspector
{
	public static function collect_enabled_modules_with_data(array $meta, array $options): array
	{
		$modules_config = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];

		$ids = [];
		foreach ($modules as $module_id => $module) {
			$module_id = sanitize_key((string)$module_id);
			if ($module_id === '') {
				continue;
			}

			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}

			$data = is_array($module['data'] ?? null) ? $module['data'] : [];
			if (self::has_meaningful_data($data)) {
				$ids[] = $module_id;
			}
		}

		sort($ids);
		return $ids;
	}

	public static function collect_enabled_modules_missing_data(array $meta, array $options): array
	{
		$modules_config = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];

		$ids = [];
		foreach ($modules_config as $module_id => $cfg) {
			$module_id = sanitize_key((string)$module_id);
			if ($module_id === '' || !self::is_geo_content_module($module_id)) {
				continue;
			}

			if (!Capabilities::is_module_available($module_id, $options)) {
				continue;
			}

			$cfg = is_array($cfg) ? $cfg : [];
			if (empty($cfg['enabled'])) {
				continue;
			}

			$module = is_array($modules[$module_id] ?? null) ? $modules[$module_id] : [];
			$data = is_array($module['data'] ?? null) ? $module['data'] : [];
			if (!self::has_meaningful_data($data)) {
				$ids[] = $module_id;
			}
		}

		sort($ids);
		return $ids;
	}

	/**
	 * @param array<int, string> $module_ids
	 * @return array<int, array{id:string,label:string,intro:string}>
	 */
	public static function module_details(array $module_ids): array
	{
		$all = Registry::all();
		$out = [];
		foreach ($module_ids as $id) {
			$id = sanitize_key((string)$id);
			if ($id === '' || !isset($all[$id])) {
				continue;
			}
			$out[] = [
				'id' => $id,
				'label' => (string)($all[$id]['label'] ?? $id),
				'intro' => (string)($all[$id]['intro'] ?? ''),
			];
		}
		return $out;
	}

	/**
	 * 面向“转化/营销”与“站长理解成本”：
	 * 不以“当前设置是否启用”为准，而以“推荐 GEO 内容模块”做一份缺失清单。
	 *
	 * @return array{
	 *   generated_ids: array<int, string>,
	 *   missing_ids: array<int, string>,
	 *   missing_detail: array<int, array{id:string,label:string,intro:string}>
	 * }
	 */
	public static function analyze_geo_modules(array $meta): array
	{
		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];
		$all = Registry::all();

		$generated = [];
		$missing = [];
		foreach ($all as $module_id => $def) {
			$module_id = sanitize_key((string)$module_id);
			if ($module_id === '' || !self::is_geo_content_module($module_id)) {
				continue;
			}

			$module = is_array($modules[$module_id] ?? null) ? $modules[$module_id] : [];
			$data = is_array($module['data'] ?? null) ? $module['data'] : [];
			if (self::has_meaningful_data($data)) {
				$generated[] = $module_id;
			} else {
				$missing[] = $module_id;
			}
		}

		sort($generated);
		sort($missing);

		return [
			'generated_ids' => $generated,
			'missing_ids' => $missing,
			'missing_detail' => self::module_details($missing),
		];
	}

	private static function is_geo_content_module(string $module_id): bool
	{
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return false;
		}
		// 仅编辑器辅助，不作为“前端 GEO 信号”。
		return $module_id !== 'slug_suggestions';
	}

	private static function has_meaningful_data($value): bool
	{
		if (is_string($value)) {
			return trim($value) !== '';
		}
		if (is_int($value) || is_float($value)) {
			return true;
		}
		if (is_bool($value)) {
			return $value === true;
		}
		if (!is_array($value)) {
			return false;
		}

		foreach ($value as $v) {
			if (self::has_meaningful_data($v)) {
				return true;
			}
		}
		return false;
	}
}

