<?php

declare(strict_types=1);

namespace ContentReady\Modules;

final class Dispatch
{
	public const MODE_OFF = 'off';
	public const MODE_ADAPTIVE = 'adaptive';
	public const MODE_LIGHTWEIGHT = 'lightweight';

	/**
	 * 轻量精选推荐组合：适用于大多数文章类型与长度，且 Token 成本更可控。
	 *
	 * @return string[]
	 */
	public static function lightweight_recommended_module_ids(): array
	{
		return [
			'summary',
			'verdict',
			'tldr',
			'toc',
			'last_updated',
		];
	}

	/**
	 * 自适应默认候选模块（“我全都要”的默认起点）。
	 *
	 * 说明：
	 * - 仅包含“内容模块”（会生成/展示的 GEO 模块）。
	 * - 排除仅编辑器辅助的模块（如 slug_suggestions），避免无意义的 Token 消耗。
	 *
	 * @return string[]
	 */
	public static function adaptive_default_module_ids(): array
	{
		$out = [];
		foreach (Registry::all_ids() as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			if (in_array($module_id, ['slug_suggestions'], true)) {
				continue;
			}
			$out[] = $module_id;
		}
		return $out;
	}

	public static function normalize_mode(string $mode, bool $is_pro): string
	{
		$mode = sanitize_key($mode);
		if (!in_array($mode, [self::MODE_OFF, self::MODE_ADAPTIVE, self::MODE_LIGHTWEIGHT], true)) {
			$mode = self::MODE_OFF;
		}

		// Free：不允许开启调度策略（只能手动配置已解锁模块）。
		if (!$is_pro && $mode !== self::MODE_OFF) {
			return self::MODE_OFF;
		}

		return $mode;
	}

	/**
	 * 返回“基于当前配置 + 轻量精选规则覆盖”后的 modules config（不写回数据库）。
	 */
	public static function apply_lightweight_modules_config(array $modules_config): array
	{
		$keep = array_flip(self::lightweight_recommended_module_ids());

		foreach (array_keys($modules_config) as $module_id) {
			$module_id = sanitize_key((string)$module_id);
			if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
				continue;
			}

			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			$enabled = isset($keep[$module_id]);

			$cfg['enabled'] = $enabled;
			$cfg['allow_ai'] = (!in_array($module_id, ['last_updated', 'toc'], true)) ? $enabled : false;
			$modules_config[$module_id] = $cfg;
		}

		return $modules_config;
	}
}

