<?php

declare(strict_types=1);

namespace ContentReady\Modules;

final class Config
{
	public static function default_modules_config(): array
	{
		$config = [];
		foreach (Registry::all() as $id => $module) {
			$requires_ai = Registry::is_content_module($id) && !in_array($id, ['last_updated', 'toc'], true);
			$config[$id] = [
				'enabled' => (bool)($module['enabled_by_default'] ?? true),
				'label' => (string)($module['label'] ?? $id),
				// 内部固定：不在设置页暴露，避免用户心智负担。
				'position' => (string)($module['position'] ?? 'bottom'), // top | after | bottom | code
				// v1：为降低用户心智负担，不单独暴露“允许生成（AI）”开关；是否参与生成由 enabled 决定。
				'allow_ai' => $requires_ai,
				'summary_excerpt' => 'off', // off | force | compat（仅 summary）
				'prompt' => '',
				'toc' => [
					'default_open' => true,
					'list_style' => 'unordered', // unordered | ordered
					'max_level' => 4, // 2-5（默认到 H4；可选到 H5）
				],
				'min_rules' => [
					'faq_min_items' => 3,
				],
			];
			if ($id !== 'toc') {
				// 仅 toc 使用：避免其他模块配置膨胀。
				unset($config[$id]['toc']);
			}
		}
		return $config;
	}

	public static function default_order(): array
	{
		return Registry::all_ids();
	}

	public static function normalize_order($order): array
	{
		$all = Registry::all_ids();
		if (!is_array($order) || $order === []) {
			return $all;
		}

		$seen = [];
		$normalized = [];
		foreach ($order as $id) {
			$id = sanitize_key((string)$id);
			if (!Registry::is_valid_id($id)) {
				continue;
			}
			if (isset($seen[$id])) {
				continue;
			}
			$seen[$id] = true;
			$normalized[] = $id;
		}

		foreach ($all as $id) {
			if (!isset($seen[$id])) {
				$normalized[] = $id;
			}
		}

		return $normalized;
	}

	public static function normalize_modules_config($value): array
	{
		$defaults = self::default_modules_config();
		if (!is_array($value)) {
			return $defaults;
		}

		$normalized = $defaults;
		foreach ($defaults as $id => $default) {
			$incoming = is_array($value[$id] ?? null) ? $value[$id] : [];
			$normalized[$id] = array_merge($default, array_intersect_key($incoming, $default));
			// position 为内部固定字段：忽略历史配置/手工篡改。
			$normalized[$id]['position'] = (string)$default['position'];
		}
		return $normalized;
	}

	public static function sanitize_modules_config($value): array
	{
		$defaults = self::default_modules_config();
		if (!is_array($value)) {
			return $defaults;
		}

		$out = $defaults;
		foreach ($defaults as $module_id => $default) {
			$incoming = is_array($value[$module_id] ?? null) ? $value[$module_id] : [];
			$out[$module_id] = self::sanitize_module_config($module_id, $incoming, $out[$module_id], $default);
		}

		return $out;
	}

	/**
	 * 仅更新表单里提交的模块项：用于 Free 隐藏 Pro 模块时，避免“未提交 == 被清空”。
	 */
	public static function sanitize_modules_config_partial($posted, $existing): array
	{
		$defaults = self::default_modules_config();
		$out = self::normalize_modules_config($existing);

		if (!is_array($posted)) {
			return $out;
		}

		foreach ($defaults as $module_id => $default) {
			if (!array_key_exists($module_id, $posted)) {
				continue;
			}
			$incoming = is_array($posted[$module_id] ?? null) ? $posted[$module_id] : [];
			$out[$module_id] = self::sanitize_module_config($module_id, $incoming, $out[$module_id], $default);
		}

		return $out;
	}

	private static function sanitize_module_config(string $module_id, array $incoming, array $base, array $default): array
	{
		$out = $base;

		$out['enabled'] = !empty($incoming['enabled']);

		$label = sanitize_text_field((string)($incoming['label'] ?? $default['label']));
		if ($label === '') {
			$label = (string)$default['label'];
		}
		$out['label'] = $label;

		if (Registry::is_content_module($module_id)) {
			$requires_ai = !in_array($module_id, ['last_updated', 'toc'], true);
			$out['allow_ai'] = $requires_ai ? $out['enabled'] : false;
			// position 为内部固定字段：不接受用户输入。
			$out['position'] = (string)$default['position'];

			$prompt = sanitize_textarea_field((string)($incoming['prompt'] ?? ''));
			if (strlen($prompt) > 4000) {
				$prompt = substr($prompt, 0, 4000);
			}
			$out['prompt'] = $prompt;

			if ($module_id === 'summary') {
				$mode = sanitize_key((string)($incoming['summary_excerpt'] ?? 'off'));
				if (!in_array($mode, ['off', 'force', 'compat'], true)) {
					$mode = 'off';
				}
				$out['summary_excerpt'] = $mode;
			} else {
				$out['summary_excerpt'] = 'off';
			}

			if ($module_id === 'toc') {
				$toc_in = is_array($incoming['toc'] ?? null) ? $incoming['toc'] : [];
				$toc_base = is_array($out['toc'] ?? null) ? $out['toc'] : (is_array($default['toc'] ?? null) ? $default['toc'] : []);
				$open = !empty($toc_in['default_open']);
				$list_style = sanitize_key((string)($toc_in['list_style'] ?? ($toc_base['list_style'] ?? 'unordered')));
				if (!in_array($list_style, ['unordered', 'ordered'], true)) {
					$list_style = 'unordered';
				}
				$max_level = isset($toc_in['max_level']) ? (int)$toc_in['max_level'] : (int)($toc_base['max_level'] ?? 4);
				$max_level = max(2, min(5, $max_level));
				$out['toc'] = [
					'default_open' => $open,
					'list_style' => $list_style,
					'max_level' => $max_level,
				];
			} else {
				unset($out['toc']);
			}
		} else {
			$out['allow_ai'] = false;
			$out['position'] = 'code';
			$out['summary_excerpt'] = 'off';
			$out['prompt'] = '';
			unset($out['toc']);
		}

		if ($module_id === 'faq') {
			$min = $incoming['min_rules']['faq_min_items'] ?? ($default['min_rules']['faq_min_items'] ?? 3);
			$min = max(1, min(20, (int)$min));
			$out['min_rules']['faq_min_items'] = $min;
		}

		return $out;
	}
}
