<?php

declare(strict_types=1);

namespace ContentReady\Modules;

/**
 * 模块自适应规则：根据内容信号判断模块是否跳过/简化。
 */
final class AdaptiveRules
{
	/**
	 * 模块规则定义。
	 * - min_words: 最小正文字数（0 表示无限制）
	 * - signal: 触发信号（null 表示无需信号，仅看字数）
	 * - priority: 预算裁剪优先级（数字越小越优先保留）
	 * - always: 始终生成，不参与自适应判断
	 */
	private const RULES = [
		'summary' => [
			'min_words' => 150,
			'signal'    => null,
			'priority'  => 1,
			'always'    => true,
		],
		'verdict' => [
			'min_words' => 300,
			'signal'    => null,
			'priority'  => 2,
			'always'    => false,
		],
		'tldr' => [
			'min_words' => 200,
			'signal'    => 'has_structure',
			'priority'  => 3,
			'always'    => false,
		],
		'references' => [
			'min_words' => 0,
			'signal'    => 'has_outbound_links',
			'priority'  => 4,
			'always'    => false,
		],
		'limitations' => [
			'min_words' => 400,
			'signal'    => null,
			'priority'  => 5,
			'always'    => false,
		],
		'faq' => [
			'min_words' => 500,
			'signal'    => 'has_question',
			'priority'  => 6,
			'always'    => false,
		],
		'table' => [
			'min_words' => 400,
			'signal'    => null,
			'priority'  => 7,
			'always'    => false,
		],
		// 以下模块始终生成
		'last_updated' => [
			'min_words' => 0,
			'signal'    => null,
			'priority'  => 0,
			'always'    => true,
		],
		'anchors' => [
			'min_words' => 0,
			'signal'    => null,
			'priority'  => 0,
			'always'    => true,
		],
		'slug_suggestions' => [
			'min_words' => 0,
			'signal'    => null,
			'priority'  => 0,
			'always'    => true,
		],
		'structured_output' => [
			'min_words' => 0,
			'signal'    => null,
			'priority'  => 0,
			'always'    => true,
		],
	];

	/**
	 * 判断模块是否应跳过。
	 *
	 * @param string $module_id 模块 ID
	 * @param array  $signals   ContentAnalyzer::analyze() 返回的信号数组
	 * @return bool 是否应跳过
	 */
	public static function should_skip(string $module_id, array $signals): bool
	{
		$rule = self::RULES[$module_id] ?? null;
		if ($rule === null) {
			// 未知模块，不跳过
			return false;
		}

		// 始终生成的模块不跳过
		if (!empty($rule['always'])) {
			return false;
		}

		$word_count = (int)($signals['word_count'] ?? 0);
		$min_words = (int)($rule['min_words'] ?? 0);
		$required_signal = $rule['signal'] ?? null;

		// 检查最小字数
		if ($min_words > 0 && $word_count < $min_words) {
			// 字数不足
			if ($required_signal === null) {
				// 无信号要求，直接跳过
				return true;
			}
			// 有信号要求，检查信号是否满足
			$has_signal = !empty($signals[$required_signal]);
			if (!$has_signal) {
				return true;
			}
		}

		// 字数满足，检查信号要求
		if ($required_signal !== null) {
			$has_signal = !empty($signals[$required_signal]);
			// references 特殊：必须有出站链接才生成
			if ($module_id === 'references' && !$has_signal) {
				return true;
			}
		}

		return false;
	}

	/**
	 * 获取输出模式：normal / short。
	 *
	 * @param string $module_id 模块 ID
	 * @param array  $signals   信号数组
	 * @return string
	 */
	public static function get_output_mode(string $module_id, array $signals): string
	{
		$word_count = (int)($signals['word_count'] ?? 0);

		// 短文章时输出精简版
		if ($word_count < 500) {
			return 'short';
		}

		return 'normal';
	}

	/**
	 * 计算输出预算（字数）。
	 *
	 * @param int $word_count 正文字数
	 * @return int 预算字数
	 */
	public static function get_budget(int $word_count): int
	{
		if ($word_count <= 500) {
			// 短文：50%
			return (int)($word_count * 0.5);
		}
		if ($word_count <= 1500) {
			// 中等：35%
			return (int)($word_count * 0.35);
		}
		// 长文：25%
		return (int)($word_count * 0.25);
	}

	/**
	 * 获取模块优先级（用于预算裁剪）。
	 *
	 * @param string $module_id
	 * @return int 优先级（数字越小越优先保留）
	 */
	public static function get_priority(string $module_id): int
	{
		return (int)(self::RULES[$module_id]['priority'] ?? 99);
	}

	/**
	 * 获取跳过原因描述。
	 *
	 * @param string $module_id
	 * @param array  $signals
	 * @return string
	 */
	public static function get_skip_reason(string $module_id, array $signals): string
	{
		$rule = self::RULES[$module_id] ?? null;
		if ($rule === null) {
			return '';
		}

		$word_count = (int)($signals['word_count'] ?? 0);
		$min_words = (int)($rule['min_words'] ?? 0);
		$required_signal = $rule['signal'] ?? null;

		if ($min_words > 0 && $word_count < $min_words) {
			return sprintf('正文字数不足（需 ≥%d 字，当前 %d 字）', $min_words, $word_count);
		}

		if ($required_signal !== null && empty($signals[$required_signal])) {
			$signal_names = [
				'has_structure'       => '结构信号（H2/H3 ≥ 2）',
				'has_question'        => '问句信号',
				'has_outbound_links'  => '出站链接',
			];
			return sprintf('缺少%s', $signal_names[$required_signal] ?? $required_signal);
		}

		return '自适应规则';
	}
}
