<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

/**
 * Pro 自动化“票据”请求：用于强制在线门禁（fail-closed）。
 *
 * 说明：
 * - ticket 仅用于“证明近期在线可用”，不承诺可离线长期使用。
 * - 当 ticket 不可用时，自动化应直接跳过并提示用户（避免离线无限跑）。
 */
final class TicketClient
{
	private const DEFAULT_TICKET_URL = 'https://cr-license.imsxx.com/v1/ticket';

	public static function ticket_url(): string
	{
		$url = (string)apply_filters('content_ready_license_ticket_url', self::DEFAULT_TICKET_URL);
		$url = trim($url);
		return esc_url_raw($url);
	}

	/**
	 * @return array{
	 *   ok:bool,
	 *   http_status:int,
	 *   data:array<string,mixed>,
	 *   error:string
	 * }
	 */
	public static function request(string $key, string $host, string $site_url, string $plugin_version, string $context): array
	{
		$url = self::ticket_url();
		if ($url === '') {
			return [
				'ok' => false,
				'http_status' => 0,
				'data' => [],
				'error' => 'missing_ticket_url',
			];
		}

                $payload = [
                        'key' => $key,
                        'host' => $host,
                        'site_url' => $site_url,
                        'plugin_version' => $plugin_version,
                        'context' => $context,
                ];

                $timeout = (int)apply_filters('content_ready_license_ticket_timeout_sec', 12);
                $timeout = max(3, min(30, $timeout));

                $resp = wp_remote_post($url, [
                        'timeout' => $timeout,
                        'headers' => [
                                'Content-Type' => 'application/json',
                        ],
                        'body' => wp_json_encode($payload),
                ]);

		if (is_wp_error($resp)) {
			return [
				'ok' => false,
				'http_status' => 0,
				'data' => [],
				'error' => sanitize_text_field((string)$resp->get_error_message()),
			];
		}

		$code = (int)wp_remote_retrieve_response_code($resp);
		$body = (string)wp_remote_retrieve_body($resp);
		$data = json_decode($body, true);
		$data = is_array($data) ? $data : [];

		if ($code < 200 || $code >= 300) {
			return [
				'ok' => false,
				'http_status' => $code,
				'data' => $data,
				'error' => $data['error'] ?? ('http_' . $code),
			];
		}

		return [
			'ok' => true,
			'http_status' => $code,
			'data' => $data,
			'error' => '',
		];
	}
}
