<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

/**
 * v1.1 License 在线校验客户端（Worker / D1）。
 *
 * 目标：
 * - 站点输入 key 后，必须“校验成功”才解锁 Pro。
 * - 采用事件触发校验（设置保存/版本更新/手动校验），失败直接返回错误状态。
 */
final class LicenseClient
{
	private const DEFAULT_VERIFY_URL = 'https://cr-license.imsxx.com/v1/verify';

	/**
	 * 在线校验接口 URL（允许通过 filter 覆盖，便于未来切换自定义域名）。
	 */
	public static function verify_url(): string
	{
		$url = (string)apply_filters('content_ready_license_verify_url', self::DEFAULT_VERIFY_URL);
		$url = trim($url);
		return esc_url_raw($url);
	}

	/**
	 * @return array{0:array,1:bool} [options, refreshed]
	 */
	public static function maybe_refresh(array $options, array $args = []): array
	{
		$force = !empty($args['force']);
		$context = sanitize_key((string)($args['context'] ?? ''));
		if ($context === '') {
			$context = 'unknown';
		}

		$license = is_array($options['license'] ?? null) ? $options['license'] : [];
		$key = trim((string)($license['key'] ?? ''));
		if ($key === '') {
			return [$options, false];
		}

		$now = time();
		$status = sanitize_key((string)($license['status'] ?? 'unverified'));

		$should = $force || $status === 'unverified';
		if (!$should) {
			return [$options, false];
		}

		$host = self::site_host();
		$site_url = (string)get_site_url();
		$plugin_version = defined('CR_VERSION') ? (string)CR_VERSION : '';

		$result = self::call_verify($key, $host, $site_url, $plugin_version, $context);
		$options['license'] = self::apply_verify_result($license, $result, $now);

		return [$options, true];
	}

	public static function site_host(): string
	{
		$url = (string)get_site_url();
		$parts = wp_parse_url($url);
		$host = is_array($parts) ? (string)($parts['host'] ?? '') : '';
		$host = strtolower(trim($host));
		$host = ltrim($host, '.');
		$host = rtrim($host, '.');
		return $host;
	}

	/**
	 * @return array{
	 *   ok:bool,
	 *   http_status:int,
	 *   data:array<string,mixed>,
	 *   error:string
	 * }
	 */
	private static function call_verify(string $key, string $host, string $site_url, string $plugin_version, string $context): array
	{
		$url = self::verify_url();
		if ($url === '') {
			return [
				'ok' => false,
				'http_status' => 0,
				'data' => [],
				'error' => 'missing_verify_url',
			];
		}

		$payload = [
			'key' => $key,
			'host' => $host,
			'site_url' => $site_url,
			'plugin_version' => $plugin_version,
			'context' => $context,
		];

		$resp = wp_remote_post($url, [
			'timeout' => 8,
			'headers' => [
				'Content-Type' => 'application/json',
			],
			'body' => wp_json_encode($payload),
		]);

		if (is_wp_error($resp)) {
			return [
				'ok' => false,
				'http_status' => 0,
				'data' => [],
				'error' => sanitize_text_field((string)$resp->get_error_message()),
			];
		}

		$code = (int)wp_remote_retrieve_response_code($resp);
		$body = (string)wp_remote_retrieve_body($resp);
		$data = json_decode($body, true);
		$data = is_array($data) ? $data : [];

		if ($code < 200 || $code >= 300) {
			return [
				'ok' => false,
				'http_status' => $code,
				'data' => $data,
				'error' => $data['error'] ?? ('http_' . $code),
			];
		}

		return [
			'ok' => true,
			'http_status' => $code,
			'data' => $data,
			'error' => '',
		];
	}

	private static function apply_verify_result(array $license, array $result, int $now): array
	{
		$next = $license;
		$next['checked_at'] = $now;
		$next['last_http_status'] = (int)($result['http_status'] ?? 0);
		$next['last_error'] = '';

		$ttl = 86400;
		$data = is_array($result['data'] ?? null) ? $result['data'] : [];
		if (isset($data['ttl_sec'])) {
			$ttl = max(15, min(604800, (int)$data['ttl_sec']));
		}
		$next['ttl_sec'] = $ttl;

		$quota_daily_limit = (int)($license['quota_daily_limit'] ?? 0);
		if (isset($data['quota_daily_limit'])) {
			$quota_daily_limit = max(0, min(100000, (int)$data['quota_daily_limit']));
		}
		$next['quota_daily_limit'] = $quota_daily_limit;

		if (empty($result['ok'])) {
			$next['last_error'] = sanitize_text_field((string)($result['error'] ?? 'request_failed'));
			$next['status'] = 'error';
			$next['plan'] = self::PLAN_FREE();
			$next['expires_at'] = $now + $ttl;
			$next['grace_until'] = 0;
			return $next;
		}

		$status = sanitize_key((string)($data['status'] ?? ''));
		$plan = sanitize_key((string)($data['plan'] ?? ''));
		$base_domain = sanitize_text_field((string)($data['base_domain'] ?? ''));
		$grant_type = sanitize_key((string)($data['grant_type'] ?? ''));
		$message = sanitize_text_field((string)($data['message'] ?? ''));

		$next['last_message'] = $message;
		$next['base_domain'] = strtolower(trim($base_domain));
		if (in_array($grant_type, ['paid', 'gift', 'promo', 'internal'], true)) {
			$next['grant_type'] = $grant_type;
		}

		if ($status === 'disabled') {
			$next['status'] = 'disabled';
			$next['plan'] = self::PLAN_FREE();
			$next['expires_at'] = $now + $ttl;
			$next['grace_until'] = 0;
			return $next;
		}

		if ($status === 'invalid_key') {
			$next['status'] = 'invalid_key';
			$next['plan'] = self::PLAN_FREE();
			$next['expires_at'] = $now + $ttl;
			$next['grace_until'] = 0;
			return $next;
		}

		if ($status === 'invalid_domain') {
			$next['status'] = 'invalid_domain';
			$next['plan'] = self::PLAN_FREE();
			$next['expires_at'] = $now + $ttl;
			$next['grace_until'] = 0;
			return $next;
		}

		if ($status !== 'active') {
			$next['status'] = $status !== '' ? $status : 'unverified';
			$next['plan'] = self::PLAN_FREE();
			$next['expires_at'] = $now + $ttl;
			$next['grace_until'] = 0;
			return $next;
		}

		if (!in_array($plan, ['pro', 'max'], true)) {
			$plan = 'pro';
		}

		$next['status'] = 'active';
		$next['plan'] = $plan;
		$next['last_ok_at'] = $now;
		$next['last_ok_plan'] = $plan;
		$next['grace_until'] = 0;
		$next['expires_at'] = $now + $ttl;

		return $next;
	}

	private static function PLAN_FREE(): string
	{
		return 'free';
	}
}
