<?php

declare(strict_types=1);

namespace ContentReady\Licensing\Capabilities;

use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Registry;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class OptionsNormalizer
{
	/**
	 * 对 options 进行“计划约束”归一化，避免 Free 通过手工 POST/篡改配置启用 Pro 能力。
	 */
	public static function normalize(array $options, bool $is_pro): array
	{
		// 结构化输出：Free 仅允许 auto（仍允许启用/停用）。
		if (!$is_pro) {
			if (!is_array($options['structured_output'] ?? null)) {
				$options['structured_output'] = [];
			}
			$options['structured_output']['mode'] = 'auto';

			// 结构化输出：Free 不允许 force（即使是 scope 级别）。
			if (is_array($options['structured_output']['scopes'] ?? null)) {
				foreach ($options['structured_output']['scopes'] as $scope => $mode) {
					$mode = sanitize_key((string)$mode);
					if ($mode === 'force') {
						$options['structured_output']['scopes'][(string)$scope] = 'auto';
					}
				}
			}
		}

		// 隐私发送范围：Free 不允许 selected_modules（细粒度）。
		if (!$is_pro) {
			if (!is_array($options['privacy'] ?? null)) {
				$options['privacy'] = [];
			}
			$scope = (string)($options['privacy']['send_scope'] ?? 'full');
			if ($scope === 'selected_modules') {
				$options['privacy']['send_scope'] = 'title_summary';
			}
		}

		// 模块配置：Free 强制关闭 Pro 内容模块。
		$modules = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		if (!$is_pro && is_array($modules)) {
			foreach (array_keys($modules) as $mid) {
				$mid = sanitize_key((string)$mid);
				if (!Registry::is_valid_id($mid) || !Registry::is_content_module($mid)) {
					continue;
				}
				if (Capabilities::is_free_content_module($mid)) {
					continue;
				}
				$cfg = is_array($modules[$mid] ?? null) ? $modules[$mid] : [];
				$cfg['enabled'] = false;
				$cfg['allow_ai'] = false;
				$modules[$mid] = $cfg;
			}
			$options['modules'] = $modules;
		}

		$page_types = PageTypesConfig::normalize_page_types($options['page_types'] ?? null);

		// 页面类型：Free 仅保留 Free 内容模块组合，且结构化策略固定为默认策略。
		if (!$is_pro) {
			$out = [];
			foreach ($page_types as $row) {
				if (!is_array($row)) {
					continue;
				}
				$mods = is_array($row['modules'] ?? null) ? $row['modules'] : [];
				$filtered = [];
				foreach ($mods as $m) {
					$m = sanitize_key((string)$m);
					if (Capabilities::is_free_content_module($m)) {
						$filtered[] = $m;
					}
				}
				$row['modules'] = $filtered;
				$row['schema_profile'] = 'article';
				$row['schema_mode'] = 'inherit';
				$out[] = $row;
			}
			$options['page_types'] = $out;
		} else {
			$options['page_types'] = $page_types;
		}

		// 自动化：Free 强制关闭。
		if (!$is_pro) {
			if (!is_array($options['generation'] ?? null)) {
				$options['generation'] = [];
			}
			$options['generation']['auto_geo_on_publish'] = false;
			$options['generation']['backfill_enabled'] = false;
			$options['generation']['adaptive_enabled'] = false;
			$options['generation']['dispatch_mode'] = 'off';
		}

		// 作者策略：Free 强制使用 WordPress 作者（站点层）。
		if (!$is_pro) {
			if (!is_array($options['site'] ?? null)) {
				$options['site'] = [];
			}
			$options['site']['author_mode'] = 'wp';
			$options['site']['brand_author_name'] = '';
			$options['site']['entity_type'] = 'organization';
			$options['site']['entity_name'] = '';
			$options['site']['entity_url'] = '';
			$options['site']['entity_same_as'] = '';
			$options['site']['entity_phone'] = '';
			$options['site']['entity_address'] = '';
			$options['site']['entity_description'] = '';
		}

		return $options;
	}
}

