<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

use ContentReady\Admin\Settings;
use ContentReady\Licensing\AutomationGate\LicenseStatusSync;
use ContentReady\Licensing\AutomationGate\MessageBuilder;
use ContentReady\Licensing\AutomationGate\NoticeStore;
use ContentReady\Licensing\AutomationGate\TicketCache;

/**
 * Pro 自动化统一门禁（在线票据，fail-closed）。
 *
 * 目标：
 * - 自动化能力（发文自动生成/回填等）执行前必须拿到“短 TTL 票据”
 * - 拿不到票据时：直接跳过，并在后台给出提示（避免离线无限跑）
 */
final class AutomationGate
{
        private const TICKET_OPTION_KEY = 'cr_pro_ticket_v1';
        private const NOTICE_OPTION_KEY = 'cr_automation_block_notice_v1';

        /**
         * @return array{allowed:bool,reason:string,message:string}
         */
        public static function ensure_ticket(string $context): array
        {
                $context = sanitize_key($context);
                if ($context === '') {
                        $context = 'automation';
                }

                $cached = TicketCache::read(self::TICKET_OPTION_KEY);
                $ticket = $cached['ticket'];
                $expires_at = $cached['expires_at'];
                if ($ticket !== '' && $expires_at - time() > 10) {
                        return [
                                'allowed' => true,
                                'reason' => '',
                                'message' => '',
                        ];
                }

                $raw = Settings::get_raw_options();
                $license = is_array($raw['license'] ?? null) ? $raw['license'] : [];
                $key = trim((string)($license['key'] ?? ''));
                if ($key === '') {
                        TicketCache::clear(self::TICKET_OPTION_KEY);
                        return [
                                'allowed' => false,
                                'reason' => 'missing_license_key',
                                'message' => MessageBuilder::message_for($context, '未填写 License Key。请先在「许可证」页填写并校验。'),
                        ];
                }

                $host = LicenseClient::site_host();
                $site_url = (string)get_site_url();
                $plugin_version = defined('CR_VERSION') ? (string)CR_VERSION : '';

                $result = TicketClient::request($key, $host, $site_url, $plugin_version, $context);
                if (empty($result['ok'])) {
                        $err = sanitize_text_field((string)($result['error'] ?? 'request_failed'));
                        TicketCache::clear(self::TICKET_OPTION_KEY);
                        return [
                                'allowed' => false,
                                'reason' => 'ticket_request_failed',
                                'message' => MessageBuilder::message_for($context, '无法获取在线票据：' . $err . '（自动化已跳过）'),
                        ];
                }

                $data = is_array($result['data'] ?? null) ? $result['data'] : [];
                $status = sanitize_key((string)($data['status'] ?? ''));
                $plan = sanitize_key((string)($data['plan'] ?? ''));
                $base_domain = sanitize_text_field((string)($data['base_domain'] ?? ''));
                $message = sanitize_text_field((string)($data['message'] ?? ''));
                $server_time = (int)($data['server_time'] ?? 0);

                if ($status !== 'active') {
                        LicenseStatusSync::apply_definitive($raw, $status, $plan, $base_domain, $message);
                        TicketCache::clear(self::TICKET_OPTION_KEY);
                        return [
                                'allowed' => false,
                                'reason' => 'ticket_denied',
                                'message' => MessageBuilder::message_for($context, MessageBuilder::status_to_human($status, $base_domain)),
                        ];
                }

                $ticket = sanitize_text_field((string)($data['ticket'] ?? ''));
                $ticket_ttl_sec = (int)($data['ticket_ttl_sec'] ?? 0);
                $ticket_expires_at = (int)($data['ticket_expires_at'] ?? 0);
                $quota_daily_limit = max(0, min(100000, (int)($data['quota_daily_limit'] ?? 0)));
                $local_expires_at = 0;

		if ($ticket_expires_at > 0 && $server_time > 0 && $ticket_expires_at > $server_time) {
			$local_expires_at = time() + ($ticket_expires_at - $server_time);
		} elseif ($ticket_ttl_sec > 0) {
			$local_expires_at = time() + max(15, min(86400, $ticket_ttl_sec));
                }

                if ($ticket === '' || $local_expires_at - time() <= 10) {
                        TicketCache::clear(self::TICKET_OPTION_KEY);
                        return [
                                'allowed' => false,
                                'reason' => 'invalid_ticket',
                                'message' => MessageBuilder::message_for($context, '票据无效或已过期（自动化已跳过）。'),
                        ];
                }

                LicenseStatusSync::apply_active($raw, $plan, $base_domain, $message, $quota_daily_limit, $local_expires_at);
                TicketCache::write(self::TICKET_OPTION_KEY, $ticket, $local_expires_at);

                return [
                        'allowed' => true,
                        'reason' => '',
                        'message' => '',
                ];
        }

        public static function record_notice(string $context, string $reason, string $message): void
        {
                NoticeStore::record(self::NOTICE_OPTION_KEY, $context, $reason, $message);
        }
}
