<?php

declare(strict_types=1);

namespace ContentReady\Generator;

use ContentReady\AI\Client;
use ContentReady\Generator\Ai\ContextBuilder;
use ContentReady\Generator\Ai\PayloadExtractor;
use ContentReady\Generator\Ai\PromptBuilder;
use ContentReady\Modules\AdaptiveRules;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\Util\ContentAnalyzer;

final class AiGenerator
{
	private Client $client;
	private array $options;
	private string $last_error = '';

	public function __construct(array $options)
	{
		$this->options = $options;
		$ai = is_array($options['ai'] ?? null) ? $options['ai'] : [];
		$this->client = new Client(
			(string)($ai['base_url'] ?? ''),
			(string)($ai['api_key'] ?? ''),
			(string)($ai['model'] ?? '')
		);
	}

	public function is_configured(): bool
	{
		return $this->client->is_configured();
	}

	public function get_last_error(): string
	{
		return $this->last_error;
	}

	public function get_target_module_ids(array $meta): array
	{
		$page_type = is_string($meta['page_type'] ?? null) ? (string)$meta['page_type'] : 'article';
		$page_types = is_array($this->options['page_types'] ?? null) ? $this->options['page_types'] : [];

		foreach ($page_types as $row) {
			if (!is_array($row)) {
				continue;
			}
			if (($row['id'] ?? null) === $page_type && is_array($row['modules'] ?? null)) {
				return array_values($row['modules']);
			}
		}

		return Registry::all_ids();
	}

	public function generate_module(int $post_id, string $module_id, array $meta): array
	{
		$post = get_post($post_id);
		if (!$post) {
			return [];
		}

		$this->last_error = '';

		// 内容信号分析
		$signals = ContentAnalyzer::analyze((string)$post->post_content, (string)get_the_title($post));

		// 自适应规则前置检测（仅当开关开启时）
		$adaptive_enabled = !empty($this->options['generation']['adaptive_enabled']);
		if ($adaptive_enabled && AdaptiveRules::should_skip($module_id, $signals)) {
			return [
				'_skipped' => true,
				'_reason'  => AdaptiveRules::get_skip_reason($module_id, $signals),
			];
		}

		$output_mode = AdaptiveRules::get_output_mode($module_id, $signals);
		$budget = AdaptiveRules::get_budget($signals['word_count']);

		$scope = (string)($this->options['privacy']['send_scope'] ?? 'full');
		$context = ContextBuilder::build($post, $scope, $output_mode, $budget);

		$ai = is_array($this->options['ai'] ?? null) ? $this->options['ai'] : [];
		$mode = sanitize_key((string)($ai['mode'] ?? 'standard'));
		if (!in_array($mode, ['quick', 'standard', 'high'], true)) {
			$mode = 'standard';
		}
		$timeout_sec = $this->resolve_timeout_sec($ai);

		$modules_cfg = ModulesConfig::normalize_modules_config($this->options['modules'] ?? null);
		$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
		$module_prompt = sanitize_textarea_field((string)($cfg['prompt'] ?? ''));

		// summary：若站点设置为“仅在摘要为空时写入（且摘要有内容则不生成）”，则直接复用现有摘要，避免不必要的 AI 调用。
		if ($module_id === 'summary') {
			$excerpt_mode = sanitize_key((string)($cfg['summary_excerpt'] ?? 'off'));
			if ($excerpt_mode === 'compat') {
				$existing_excerpt = trim((string)get_the_excerpt($post));
				if ($existing_excerpt !== '') {
					return ['text' => $existing_excerpt];
				}
			}
		}

		$system = "你是 WordPress 内容编辑助手。你必须只输出 JSON，不要输出任何解释性文字。";
		$user = PromptBuilder::build($module_id, $mode, $module_prompt, $context);

		$res = $this->client->chat_json($system, $user, $timeout_sec);
		if (empty($res['ok'])) {
			$this->last_error = is_string($res['error'] ?? null) ? (string)$res['error'] : 'ai_request_failed';
			return [];
		}

		$json = is_array($res['json'] ?? null) ? $res['json'] : [];
		return PayloadExtractor::extract($module_id, $json);
	}

	private function resolve_timeout_sec(array $ai): int
	{
		$timeout = (int)($ai['request_timeout_sec'] ?? 90);
		return max(20, min(300, $timeout));
	}
}
