<?php

declare(strict_types=1);

namespace ContentReady\Generator\Ai;

final class PromptBuilder
{
	public static function build(string $module_id, string $mode, string $module_prompt, array $context): string
	{
		$budget = (int)($context['_budget'] ?? 0);
		$output_mode = (string)($context['_output_mode'] ?? 'normal');

		// 移除内部元数据，不传给 AI
		unset($context['_budget'], $context['_output_mode']);

		$base_rules = [
			"输出必须是合法 JSON。",
			"字段必须严格匹配要求的 schema，不能额外输出字段。",
			"尽量使用简体中文。",
			"允许使用基础排版 HTML：p、br、ul、ol、li、a、strong、em、code、pre；禁止任何自定义属性与脚本。",
		];

		// 预算约束
		if ($budget > 0) {
			$base_rules[] = "本文较短，所有模块输出总预算约 {$budget} 字，请控制本模块输出长度。";
		}

		// 输出模式约束
		if ($output_mode === 'short') {
			$base_rules[] = "文章较短，请输出精简版本，减少条目数量。";
		}

		if ($mode === 'quick') {
			$base_rules[] = "尽量简洁，优先短句与要点。";
		} elseif ($mode === 'high') {
			$base_rules[] = "尽量更全面，信息密度更高但避免废话。";
		}

		$ctx_json = wp_json_encode($context, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
		if (!is_string($ctx_json)) {
			$ctx_json = '{}';
		}

		$tldr_n = $mode === 'quick' ? '3' : ($mode === 'high' ? '5-7' : '3-5');
		$lim_n = $mode === 'quick' ? '3' : ($mode === 'high' ? '5-7' : '3-5');
		$faq_n = $mode === 'quick' ? '3-4' : ($mode === 'high' ? '5-8' : '3-6');
		$sum_len = $mode === 'quick' ? '80-150' : ($mode === 'high' ? '150-260' : '120-200');
		$ver_sent = $output_mode === 'short' ? '1' : '1-2';
		$ver_len = $mode === 'quick' ? '20-60' : ($mode === 'high' ? '60-120' : '30-90');

		$extra = trim($module_prompt);
		$extra_block = '';
		if ($extra !== '') {
			$extra_block = "\n\n补充要求（来自站点设置的「模块提示词」；必须仍然严格遵守 schema）：\n" . $extra;
		}

		switch ($module_id) {
			case 'summary':
				return implode("\n", $base_rules) . "\n\n生成一段文章摘要（用于摘要/Excerpt），长度约 " . $sum_len . " 字。\n要求：\n- 不包含标题编号，不要分点\n- 不要出现“本文/本篇”自指语\n- 重点概括“文章讲了什么/解决什么问题/读者能获得什么”，避免写成核心结论\n\nschema:\n{ \"text\": \"...\" }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'verdict':
				return implode("\n", $base_rules) . "\n\n生成核心结论（" . $ver_sent . " 句话，长度约 " . $ver_len . " 字）。\n要求：\n- 给出明确结论/建议，回答读者最关心的“结论是什么/该怎么做”\n- 避免复述摘要，不要写成 TL;DR 列表\n- 不要出现“本文/本篇”自指语\n\nschema:\n{ \"text\": \"...\" }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'tldr':
				return implode("\n", $base_rules) . "\n\n生成 " . $tldr_n . " 条快速要点。\n要求：\n- 每条尽量短，信息密度高\n- 避免重复“核心结论”的原句，尽量从不同角度拆解（事实/步骤/注意事项/行动点）\n\nschema:\n{ \"items\": [\"...\", \"...\"] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'limitations':
				return implode("\n", $base_rules) . "\n\n生成 " . $lim_n . " 条适用范围与限制。\nschema:\n{ \"items\": [\"...\", \"...\"] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'faq':
				return implode("\n", $base_rules) . "\n\n生成 " . $faq_n . " 条 FAQ。\nschema:\n{ \"items\": [{\"q\": \"...\", \"a\": \"...\"}] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'references':
				return implode("\n", $base_rules) . "\n\n如上下文中出现可作为来源的链接或明确来源信息，请输出参考资料；否则输出空数组。\nschema:\n{ \"items\": [{\"title\": \"\", \"url\": \"\"}] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'table':
				return implode("\n", $base_rules) . "\n\n判断该文章是否值得生成一个信息密度高的表格（便于读者快速对比/归纳）。\n- 如果不值得生成：返回空表格（columns=[], rows=[]），并在 notes 写明“本文信息密度不足以生成表格表单”。\n- 如果值得生成：输出 3-6 列、3-10 行，单元格尽量简短。\nschema:\n{ \"caption\": \"\", \"columns\": [\"\"], \"rows\": [[\"...\"]], \"notes\": \"\" }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'slug_suggestions':
				return implode("\n", $base_rules) . "\n\n为该文章推荐 5 个英文 slug（用于 WordPress post_name）。\n要求：\n- 全小写，仅允许 a-z、0-9、连字符 '-'\n- 不要输出中文或十六进制编码\n- 每个 slug 2-6 个词，长度 20-60\n- 避免无意义词（the/a/of 等）\n\nschema:\n{ \"suggestions\": [\"a-b-c\"] }\n\ncontext:\n" . $ctx_json . $extra_block;
			default:
				return implode("\n", $base_rules) . "\n\n对该模块输出空对象。\nschema:\n{ }\n\ncontext:\n" . $ctx_json . $extra_block;
		}
	}
}

