<?php

declare(strict_types=1);

namespace ContentReady\Frontend\StructuredData;

use ContentReady\Admin\Settings;
use ContentReady\Meta\MetaStore;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class Preview
{
	/**
	 * 构建结构化输出预览（用于编辑器侧展示/排查）。
	 *
	 * Args:
	 *   $post_id: 文章 ID
	 *
	 * Returns:
	 *   array: 预览结果
	 */
	public static function build(int $post_id): array
	{
		$post = get_post($post_id);
		if (!$post) {
			return [
				'post_id' => $post_id,
				'objects' => [],
				'would_output' => false,
				'blocked_reason' => 'post_not_found',
			];
		}

		$options = Settings::get_options();
		$structured = $options['structured_output'] ?? [];
		$enabled = !empty($structured['enabled']);
		$mode = (string)($structured['mode'] ?? 'auto'); // auto | force | off（全局默认）

		$scope = 'singular_cpt';
		if ($post->post_type === 'post') {
			$scope = 'singular_post';
		} elseif ($post->post_type === 'page') {
			$scope = 'singular_page';
		}

		$meta = MetaStore::get($post_id);
		$page_type = (string)($meta['page_type'] ?? 'article');
		$effective_mode = ModeResolver::effective_mode_for_post($options, $scope, $meta);
		$schema_strategy = PageTypesConfig::schema_strategy_for($page_type, $options);

		$compat = is_array($meta['compat'] ?? null) ? $meta['compat'] : [];
		$checked_at = (int)($compat['checked_at'] ?? 0);
		$conflict = !empty($compat['jsonld_conflict']);
		$last_error = sanitize_text_field((string)($compat['last_error'] ?? ''));

		$would_output = false;
		$blocked_reason = '';

		$objects = ObjectsBuilder::build_for_preview($options, $post, $meta);

		if (!$enabled || $effective_mode === 'off') {
			$blocked_reason = 'structured_disabled';
		} elseif ($objects === []) {
			$blocked_reason = 'no_objects_enabled';
		} elseif ($effective_mode === 'auto' && $checked_at <= 0) {
			$blocked_reason = 'auto_not_checked_yet';
		} elseif ($effective_mode === 'auto' && $last_error !== '') {
			$blocked_reason = 'auto_check_failed';
		} elseif ($effective_mode === 'auto' && $conflict) {
			$blocked_reason = 'auto_conflict_detected';
		} else {
			$would_output = true;
		}

		return [
			'post_id' => $post_id,
			'objects' => $objects,
			'would_output' => $would_output,
			'blocked_reason' => $blocked_reason,
			'structured_output' => [
				'enabled' => $enabled,
				'mode' => $mode,
				'scope' => $scope,
				'effective_mode' => $effective_mode,
			],
			'page_type' => $page_type,
			'schema_strategy' => [
				'profile' => (string)($schema_strategy['profile'] ?? 'article'),
				'mode' => (string)($schema_strategy['mode'] ?? 'inherit'),
			],
			'compat' => [
				'checked_at' => $checked_at,
				'jsonld_conflict' => $conflict,
				'last_error' => $last_error,
			],
		];
	}
}

