<?php

declare(strict_types=1);

namespace ContentReady\Frontend\StructuredData;

use ContentReady\Frontend\Structured\JsonLdBuilder;
use ContentReady\Meta\MetaStore;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class ObjectsBuilder
{
	public static function build_for_current_request(array $options, string $scope): array
	{
		$objects = [];

		$want_site_graph = ModeResolver::object_enabled($options, 'site_graph');
		if ($want_site_graph) {
			$objects[] = JsonLdBuilder::build_site_graph($options);
		}

		if (ModeResolver::object_enabled($options, 'webpage')) {
			$objects[] = JsonLdBuilder::build_webpage($options, $scope, $want_site_graph);
		}

		$post_id = is_singular() ? (int)get_queried_object_id() : 0;
		if ($post_id > 0) {
			$meta = MetaStore::get($post_id);
			$page_type = (string)($meta['page_type'] ?? 'article');
			$strategy = PageTypesConfig::schema_strategy_for($page_type, $options);
			$schema_profile = (string)($strategy['profile'] ?? 'article');

			$post = get_post($post_id);
			if ($post) {
				if (ModeResolver::object_enabled($options, 'article')) {
					$primary = JsonLdBuilder::build_primary($post, $options, $meta, $schema_profile);
					if (is_array($primary) && $primary !== []) {
						$objects[] = $primary;
					}
				}

				if (ModeResolver::object_enabled($options, 'faq') && self::profile_supports_faq($schema_profile)) {
					$faq_items = $meta['modules']['faq']['data']['items'] ?? null;
					if (is_array($faq_items) && $faq_items !== []) {
						$faq = JsonLdBuilder::build_faq($faq_items, get_permalink($post));
						if ($faq) {
							$objects[] = $faq;
						}
					}
				}
			}
		}

		/**
		 * 允许第三方追加/删减 JSON-LD objects。
		 *
		 * @param array  $objects objects 数组
		 * @param string $scope   当前 scope
		 * @param array  $options 全量 options（不含敏感信息的安全性由调用方自行确保）
		 */
		$objects = apply_filters('content_ready_jsonld_objects', $objects, $scope, $options);
		return is_array($objects) ? $objects : [];
	}

	public static function build_for_preview(array $options, \WP_Post $post, array $meta): array
	{
		$objects = [];

		$want_site_graph = ModeResolver::object_enabled($options, 'site_graph');
		if ($want_site_graph) {
			$objects[] = JsonLdBuilder::build_site_graph($options);
		}

		if (ModeResolver::object_enabled($options, 'webpage')) {
			$objects[] = JsonLdBuilder::build_webpage_for_post($options, $post, $want_site_graph);
		}

		$page_type = (string)($meta['page_type'] ?? 'article');
		$strategy = PageTypesConfig::schema_strategy_for($page_type, $options);
		$schema_profile = (string)($strategy['profile'] ?? 'article');
		if (ModeResolver::object_enabled($options, 'article')) {
			$primary = JsonLdBuilder::build_primary($post, $options, $meta, $schema_profile);
			if (is_array($primary) && $primary !== []) {
				$objects[] = $primary;
			}
		}

		if (ModeResolver::object_enabled($options, 'faq') && self::profile_supports_faq($schema_profile)) {
			$faq_items = $meta['modules']['faq']['data']['items'] ?? null;
			if (is_array($faq_items) && $faq_items !== []) {
				$faq = JsonLdBuilder::build_faq($faq_items, get_permalink($post));
				if ($faq) {
					$objects[] = $faq;
				}
			}
		}

		return $objects;
	}

	private static function profile_supports_faq(string $schema_profile): bool
	{
		$schema_profile = sanitize_key($schema_profile);
		return in_array($schema_profile, ['article', 'review', 'product', 'service', 'howto'], true);
	}
}

