<?php

declare(strict_types=1);

namespace ContentReady\Frontend;

use ContentReady\Admin\Settings;
use ContentReady\Frontend\StructuredData\JsonLdHtml;
use ContentReady\Frontend\StructuredData\ModeResolver;
use ContentReady\Frontend\StructuredData\ObjectsBuilder;
use ContentReady\Frontend\StructuredData\OutputDecider;
use ContentReady\Frontend\StructuredData\Preview;
use ContentReady\Frontend\StructuredData\ScopeDetector;
use ContentReady\Frontend\StructuredData\StatusRecorder;
use ContentReady\Meta\MetaStore;

final class StructuredData
{
	private static $head_buffer_started = false;
	private static $head_buffer_level = 0;

	public static function register(): void
	{
		// 使用 head 输出缓冲：用于在输出前探测页面中是否已有第三方 JSON-LD。
		add_action('wp_head', [self::class, 'start_head_buffer'], 0);
		add_action('wp_head', [self::class, 'flush_head_buffer'], 99999);
	}

	/**
	 * 结构化输出预览（用于编辑器侧展示/排查）。
	 *
	 * 返回：
	 * - objects: JSON-LD 对象数组（即使 auto 被阻断，也会尽量生成供预览）
	 * - would_output: 当前设置 + compat 状态下，前端是否会实际输出
	 * - blocked_reason: 仅当 would_output=false 时给出阻断原因 code
	 */
	public static function preview(int $post_id): array
	{
		return Preview::build($post_id);
	}

	public static function start_head_buffer(): void
	{
		if (is_admin()) {
			return;
		}
		if (self::$head_buffer_started) {
			return;
		}

		self::$head_buffer_started = true;
		self::$head_buffer_level = ob_get_level();
		ob_start();
	}

	public static function flush_head_buffer(): void
	{
		if (!self::$head_buffer_started) {
			return;
		}

		// 若缓冲层级异常（可能被其他代码提前结束），放弃注入以避免破坏输出。
		if (ob_get_level() <= self::$head_buffer_level) {
			self::$head_buffer_started = false;
			self::$head_buffer_level = 0;
			return;
		}

		$head = (string)ob_get_clean();
		self::$head_buffer_started = false;
		self::$head_buffer_level = 0;

		$options = Settings::get_options();
		$scope = ScopeDetector::detect();
		$mode = ModeResolver::effective_mode($options, $scope);
		if (is_singular()) {
			$post_id = (int)get_queried_object_id();
			if ($post_id > 0) {
				$meta = MetaStore::get($post_id);
				$mode = ModeResolver::effective_mode_for_post($options, $scope, $meta);
			}
		}

		$other_scripts = JsonLdHtml::count_other_jsonld_scripts($head);

		$objects = ObjectsBuilder::build_for_current_request($options, $scope);

		$blocked_reason = '';
		$would_output = OutputDecider::should_output_for_current_request($options, $scope, $mode, $other_scripts, $objects, $blocked_reason);

		StatusRecorder::record_scope_status($scope, $mode, $would_output, $blocked_reason, $other_scripts);

		echo $head;

		if (!$would_output || $objects === []) {
			return;
		}

		echo JsonLdHtml::render_objects($objects);
	}
}
