<?php

declare(strict_types=1);

namespace ContentReady\Frontend\Structured\JsonLd;

final class WebPageBuilder
{
	public static function build_current(array $options, string $scope, bool $with_site_graph): array
	{
		$post_id = is_singular() ? (int)get_queried_object_id() : 0;
		$url = Resolver::resolve_current_canonical_url($scope, $post_id);
		if ($url === '') {
			$url = home_url('/');
		}

		$name = '';
		if (is_singular() && $post_id > 0) {
			$name = (string)get_the_title($post_id);
		}
		if ($name === '') {
			$name = (string)wp_get_document_title();
		}
		$name = trim(wp_strip_all_tags($name));

		$desc = Resolver::resolve_current_description($scope, $post_id);
		$desc = trim(wp_strip_all_tags($desc));

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'WebPage',
			'@id' => (strpos($url, '#') === false) ? ($url . '#webpage') : $url,
			'url' => $url,
			'name' => $name,
		];

		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$locale = Resolver::resolve_site_locale($site);
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}
		if ($desc !== '') {
			$obj['description'] = $desc;
		}

		if ($with_site_graph) {
			$site_url = Resolver::site_url_base();
			if ($site_url !== '') {
				$obj['isPartOf'] = ['@id' => $site_url . '#website'];
				$obj['publisher'] = ['@id' => EntityResolver::entity_id($site)];
			}
		}

		return $obj;
	}

	public static function build_for_post(array $options, \WP_Post $post, bool $with_site_graph): array
	{
		$url = get_permalink($post);
		$url = is_string($url) ? esc_url_raw($url) : '';
		if ($url === '') {
			$url = home_url('/');
		}

		$name = trim(wp_strip_all_tags((string)get_the_title($post)));
		if ($name === '') {
			$name = trim(wp_strip_all_tags((string)wp_get_document_title()));
		}

		$desc = '';
		if (isset($post->post_excerpt) && (string)$post->post_excerpt !== '') {
			$desc = (string)$post->post_excerpt;
		}
		$desc = trim(wp_strip_all_tags($desc));

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'WebPage',
			'@id' => (strpos($url, '#') === false) ? ($url . '#webpage') : $url,
			'url' => $url,
			'name' => $name,
		];

		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$locale = Resolver::resolve_site_locale($site);
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}
		if ($desc !== '') {
			$obj['description'] = $desc;
		}

		if ($with_site_graph) {
			$site_url = Resolver::site_url_base();
			if ($site_url !== '') {
				$obj['isPartOf'] = ['@id' => $site_url . '#website'];
				$obj['publisher'] = ['@id' => EntityResolver::entity_id($site)];
			}
		}

		return $obj;
	}
}

