<?php

declare(strict_types=1);

namespace ContentReady\Frontend\Structured\JsonLd;

use ContentReady\Util\Locale;

final class Resolver
{
	public static function site_url_base(): string
	{
		$url = home_url('/');
		$url = is_string($url) ? $url : '';
		$url = trim($url);
		return $url !== '' ? trailingslashit(esc_url_raw($url)) : '';
	}

	public static function resolve_site_locale(array $site): string
	{
		$locale = Locale::normalize_bcp47((string)($site['locale'] ?? ''));
		if ($locale === '') {
			$locale = Locale::wp_locale_to_bcp47((string)get_locale());
		}
		return $locale;
	}

	public static function resolve_current_canonical_url(string $scope, int $post_id): string
	{
		$scope = sanitize_key((string)$scope);

		if (is_singular() && $post_id > 0) {
			$url = get_permalink($post_id);
			return is_string($url) ? esc_url_raw($url) : '';
		}

		if ($scope === 'home') {
			// 如果首页是“文章列表页”（page_for_posts），尽量给出该页面 permalink。
			if (is_home() && !is_front_page()) {
				$pid = (int)get_option('page_for_posts');
				if ($pid > 0) {
					$url = get_permalink($pid);
					return is_string($url) ? esc_url_raw($url) : '';
				}
			}
			return esc_url_raw(home_url('/'));
		}

		if ($scope === 'taxonomy') {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$url = get_term_link($term);
				return is_string($url) ? esc_url_raw($url) : '';
			}
		}

		if ($scope === 'search') {
			$q = get_search_query(false);
			$url = get_search_link($q);
			return is_string($url) ? esc_url_raw($url) : '';
		}

		return '';
	}

	public static function resolve_current_description(string $scope, int $post_id): string
	{
		$scope = sanitize_key((string)$scope);

		if ($scope === 'home') {
			$desc = get_bloginfo('description');
			return is_string($desc) ? $desc : '';
		}

		if (is_singular() && $post_id > 0) {
			$post = get_post($post_id);
			if ($post && !empty($post->post_excerpt)) {
				return (string)$post->post_excerpt;
			}
			return '';
		}

		if ($scope === 'taxonomy') {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				return (string)$term->description;
			}
		}

		if ($scope === 'search') {
			$q = trim((string)get_search_query(false));
			if ($q !== '') {
				return '搜索结果：' . $q;
			}
		}

		return '';
	}

	public static function resolve_publisher_logo_url(array $site): string
	{
		$explicit = esc_url_raw((string)($site['logo_url'] ?? ''));
		if ($explicit !== '') {
			return $explicit;
		}

		$icon = get_site_icon_url();
		if (is_string($icon) && $icon !== '') {
			return $icon;
		}

		$path = rtrim((string)ABSPATH, "/\\") . DIRECTORY_SEPARATOR . 'favicon.ico';
		if (is_file($path)) {
			return home_url('/favicon.ico');
		}

		$doc_root = isset($_SERVER['DOCUMENT_ROOT']) ? (string)$_SERVER['DOCUMENT_ROOT'] : '';
		if ($doc_root !== '') {
			$path2 = rtrim($doc_root, "/\\") . DIRECTORY_SEPARATOR . 'favicon.ico';
			if (is_file($path2)) {
				return home_url('/favicon.ico');
			}
		}

		return '';
	}
}

