<?php

declare(strict_types=1);

namespace ContentReady\Frontend\Structured\JsonLd;

use ContentReady\Util\HtmlHeadings;

final class PrimaryBuilder
{
	public static function build(\WP_Post $post, array $options, array $meta, string $schema_profile): array
	{
		$schema_profile = sanitize_key($schema_profile);
		if (!in_array($schema_profile, ['article', 'review', 'product', 'service', 'howto'], true)) {
			$schema_profile = 'article';
		}

		if ($schema_profile === 'article') {
			return ArticleBuilder::build($post, $options, $meta);
		}

		if ($schema_profile === 'review') {
			return self::build_review($post, $options, $meta);
		}

		if ($schema_profile === 'product') {
			return self::build_product($post, $options);
		}

		if ($schema_profile === 'service') {
			return self::build_service($post, $options);
		}

		return self::build_howto($post, $options);
	}

	private static function build_review(\WP_Post $post, array $options, array $meta): array
	{
		$obj = ArticleBuilder::build($post, $options, $meta);
		$title = (string)get_the_title($post);
		$obj['@type'] = 'Review';
		$obj['name'] = $title;
		$obj['itemReviewed'] = [
			'@type' => 'Thing',
			'name' => $title,
		];
		return $obj;
	}

	private static function build_product(\WP_Post $post, array $options): array
	{
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$url = self::post_url($post);
		$publisher_name = EntityResolver::entity_name($site);

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'Product',
			'@id' => (strpos($url, '#') === false) ? ($url . '#product') : $url,
			'url' => $url,
			'name' => (string)get_the_title($post),
			'description' => self::post_description($post),
			'brand' => [
				'@type' => 'Brand',
				'name' => $publisher_name,
			],
		];

		$locale = Resolver::resolve_site_locale($site);
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}

		return $obj;
	}

	private static function build_service(\WP_Post $post, array $options): array
	{
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$url = self::post_url($post);
		$provider = EntityResolver::build_primary_entity($site);

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'Service',
			'@id' => (strpos($url, '#') === false) ? ($url . '#service') : $url,
			'url' => $url,
			'name' => (string)get_the_title($post),
			'description' => self::post_description($post),
			'provider' => $provider,
		];

		$locale = Resolver::resolve_site_locale($site);
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}

		return $obj;
	}

	private static function build_howto(\WP_Post $post, array $options): array
	{
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$url = self::post_url($post);
		$steps = self::howto_steps((string)$post->post_content);

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'HowTo',
			'@id' => (strpos($url, '#') === false) ? ($url . '#howto') : $url,
			'url' => $url,
			'name' => (string)get_the_title($post),
			'description' => self::post_description($post),
			'step' => $steps,
		];

		$locale = Resolver::resolve_site_locale($site);
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}

		return $obj;
	}

	private static function post_url(\WP_Post $post): string
	{
		$url = get_permalink($post);
		$url = is_string($url) ? esc_url_raw($url) : '';
		if ($url === '') {
			$url = esc_url_raw(home_url('/'));
		}
		return $url;
	}

	private static function post_description(\WP_Post $post): string
	{
		$excerpt = trim((string)$post->post_excerpt);
		if ($excerpt !== '') {
			return $excerpt;
		}

		return (string)wp_trim_words((string)wp_strip_all_tags((string)$post->post_content), 60, '…');
	}

	/**
	 * @return array<int, array{ '@type':string, name:string, text:string }>
	 */
	private static function howto_steps(string $html): array
	{
		$headings = HtmlHeadings::extract($html, 2, 4);
		$out = [];
		$index = 0;
		foreach ($headings as $row) {
			$text = trim((string)($row['text'] ?? ''));
			if ($text === '') {
				continue;
			}
			$index++;
			$out[] = [
				'@type' => 'HowToStep',
				'name' => '步骤 ' . $index,
				'text' => $text,
			];
			if ($index >= 8) {
				break;
			}
		}

		if ($out !== []) {
			return $out;
		}

		return [[
			'@type' => 'HowToStep',
			'name' => '步骤 1',
			'text' => '请按正文步骤执行。',
		]];
	}
}
