<?php

declare(strict_types=1);

namespace ContentReady\Frontend\Structured\JsonLd;

final class EntityResolver
{
	public static function entity_type(array $site): string
	{
		$type = sanitize_key((string)($site['entity_type'] ?? 'organization'));
		if (!in_array($type, ['organization', 'person', 'local_business'], true)) {
			return 'organization';
		}
		return $type;
	}

	public static function entity_name(array $site): string
	{
		$name = trim((string)($site['entity_name'] ?? ''));
		if ($name !== '') {
			return $name;
		}

		$name = trim((string)($site['display_name'] ?? ''));
		if ($name !== '') {
			return $name;
		}

		return (string)get_bloginfo('name');
	}

	public static function entity_url(array $site): string
	{
		$url = esc_url_raw((string)($site['entity_url'] ?? ''));
		if ($url !== '') {
			return $url;
		}

		$base = Resolver::site_url_base();
		if ($base !== '') {
			return $base;
		}

		return esc_url_raw(home_url('/'));
	}

	public static function entity_id(array $site): string
	{
		$base = self::entity_url($site);
		if ($base === '') {
			$base = esc_url_raw(home_url('/'));
		}
		if ($base === '') {
			$base = '/';
		}

		return (strpos($base, '#') === false) ? ($base . '#entity') : $base;
	}

	public static function build_primary_entity(array $site): array
	{
		$type = self::entity_type($site);
		$type_map = [
			'organization' => 'Organization',
			'person' => 'Person',
			'local_business' => 'LocalBusiness',
		];

		$entity = [
			'@type' => $type_map[$type] ?? 'Organization',
			'@id' => self::entity_id($site),
			'name' => self::entity_name($site),
			'url' => self::entity_url($site),
		];

		$description = trim((string)($site['entity_description'] ?? ''));
		if ($description !== '') {
			$entity['description'] = $description;
		}

		$same_as = self::normalize_same_as((string)($site['entity_same_as'] ?? ''));
		if ($same_as !== []) {
			$entity['sameAs'] = $same_as;
		}

		$logo = Resolver::resolve_publisher_logo_url($site);
		if ($logo !== '' && in_array($type, ['organization', 'local_business'], true)) {
			$entity['logo'] = [
				'@type' => 'ImageObject',
				'url' => $logo,
			];
		}

		if ($type === 'local_business') {
			$phone = trim((string)($site['entity_phone'] ?? ''));
			if ($phone !== '') {
				$entity['telephone'] = $phone;
			}

			$address = trim((string)($site['entity_address'] ?? ''));
			if ($address !== '') {
				$entity['address'] = $address;
			}
		}

		return $entity;
	}

	/**
	 * @return string[]
	 */
	private static function normalize_same_as(string $raw): array
	{
		$raw = str_replace(["\r\n", "\r"], "\n", $raw);
		$items = preg_split('/[\n,]+/', $raw);
		$items = is_array($items) ? $items : [];

		$out = [];
		foreach ($items as $item) {
			$url = esc_url_raw(trim((string)$item));
			if ($url === '') {
				continue;
			}
			$out[$url] = true;
		}

		return array_keys($out);
	}
}
