<?php

declare(strict_types=1);

namespace ContentReady\Frontend\Structured\JsonLd;

use ContentReady\Licensing\Capabilities;

final class ArticleBuilder
{
	public static function build(\WP_Post $post, array $options, array $meta): array
	{
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$publisher_name = EntityResolver::entity_name($site);
		$publisher = EntityResolver::build_primary_entity($site);

		$is_pro = Capabilities::is_pro($options);
		$author_mode = 'wp';
		$brand_author_name = '';
		if ($is_pro) {
			$site_author_mode = (string)($site['author_mode'] ?? 'wp');
			if (!in_array($site_author_mode, ['wp', 'brand'], true)) {
				$site_author_mode = 'wp';
			}

			$override = is_array($meta['author'] ?? null) ? $meta['author'] : [];
			$override_mode = sanitize_key((string)($override['mode'] ?? 'inherit'));
			if (!in_array($override_mode, ['inherit', 'wp', 'brand'], true)) {
				$override_mode = 'inherit';
			}

			$author_mode = $override_mode === 'inherit' ? $site_author_mode : $override_mode;

			$brand_author_name = trim((string)($override['name'] ?? ''));
			if ($brand_author_name === '') {
				$brand_author_name = trim((string)($site['brand_author_name'] ?? ''));
			}
			if ($brand_author_name === '') {
				$brand_author_name = $publisher_name;
			}
		}

		$author_obj = null;
		if ($author_mode === 'brand') {
			$author_obj = [
				'@type' => 'Organization',
				'name' => $brand_author_name,
			];
		} else {
			$author_name = '';
			$author = get_user_by('id', (int)$post->post_author);
			if ($author && isset($author->display_name)) {
				$author_name = (string)$author->display_name;
			}
			if ($author_name !== '') {
				$author_obj = [
					'@type' => 'Person',
					'name' => $author_name,
				];
			}
		}

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'Article',
			'mainEntityOfPage' => [
				'@type' => 'WebPage',
				'@id' => get_permalink($post),
			],
			'headline' => get_the_title($post),
			'datePublished' => get_the_date('c', $post),
			'dateModified' => get_post_modified_time('c', false, $post),
			'publisher' => $publisher,
		];

		$locale = Resolver::resolve_site_locale($site);
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}

		if (is_array($author_obj)) {
			$obj['author'] = $author_obj;
		}

		return $obj;
	}
}

