<?php

declare(strict_types=1);

namespace ContentReady\Frontend;

use ContentReady\Admin\Settings;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Registry;
use ContentReady\Util\Anchors;

final class Renderer
{
	public static function register(): void
	{
		add_filter('the_content', [self::class, 'inject_modules'], 15);
		add_action('wp_enqueue_scripts', [self::class, 'enqueue_assets']);
	}

	public static function enqueue_assets(): void
	{
		$options = Settings::get_options();
		if (empty($options['frontend']['css_enabled'])) {
			return;
		}

		wp_enqueue_style(
			'content-ready-frontend',
			CR_PLUGIN_URL . 'assets/frontend.css',
			[],
			CR_VERSION
		);
	}

	public static function inject_modules(string $content): string
	{
		if (is_admin() || !is_singular() || !in_the_loop() || !is_main_query()) {
			return $content;
		}

		$options = Settings::get_options();
		if (empty($options['frontend']['inject_enabled'])) {
			return $content;
		}

		$post_id = get_the_ID();
		if (!$post_id) {
			return $content;
		}

		$meta = MetaStore::get((int)$post_id);
		$modules_cfg = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		// 模块顺序由插件内部固定（不从 options 读取，避免用户配置造成体验分裂）。
		$order = Registry::all_ids();

		$anchors_cfg = is_array($modules_cfg['anchors'] ?? null) ? $modules_cfg['anchors'] : [];
		$anchors_enabled = !empty($anchors_cfg['enabled']);

		if ($anchors_enabled) {
			$content = self::inject_heading_anchors($content, $meta);
		}

		$top = '';
		$after = '';
		$bottom = '';
		$table_html = '';
		$table_data = [];

		foreach ($order as $module_id) {
			$module_id = sanitize_key((string)$module_id);
			if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
				continue;
			}
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}
			$data = $meta['modules'][$module_id]['data'] ?? [];
			if (!is_array($data) || $data === []) {
				continue;
			}

			$anchor = $anchors_enabled ? Anchors::get_module_anchor($meta, $module_id) : '';
			$html = ModuleRenderer::render($module_id, (string)($cfg['label'] ?? $module_id), $data, $anchor);
			if ($html === '') {
				continue;
			}

			if ($module_id === 'table') {
				$table_html = $html;
				$table_data = $data;
				continue;
			}

			$pos = (string)($cfg['position'] ?? 'bottom');
			if ($pos === 'top') {
				$top .= $html;
			} elseif ($pos === 'after') {
				$after .= $html;
			} else {
				$bottom .= $html;
			}
		}

		if ($table_html !== '') {
			$content = TablePlacement::inject($content, $table_html, $table_data);
		}

		return $top . $content . $after . $bottom;
	}

	private static function inject_heading_anchors(string $html, array $meta): string
	{
		if ($html === '') {
			return $html;
		}

		$idx = 0;
		$used = [
			'verdict' => true,
			'tldr' => true,
			'limitations' => true,
			'faq' => true,
			'references' => true,
			'updated' => true,
		];

		return (string)preg_replace_callback(
			'~<(h[2-6])\\b([^>]*)>(.*?)</\\1>~is',
			static function (array $m) use (&$idx, &$used, $meta): string {
				$tag = isset($m[1]) ? (string)$m[1] : 'h2';
				$attrs = isset($m[2]) ? (string)$m[2] : '';
				$inner = isset($m[3]) ? (string)$m[3] : '';

				if (preg_match('~\\bid\\s*=~i', $attrs) === 1) {
					$idx++;
					return '<' . $tag . $attrs . '>' . $inner . '</' . $tag . '>';
				}

				$a = Anchors::get_heading_anchor($meta, $idx);
				$id = (string)($a['id'] ?? '');
				$legacy_id = (string)($a['legacy_id'] ?? '');
				if ($id === '') {
					$text = trim(wp_strip_all_tags($inner));
					$id = Anchors::sanitize_anchor_id((string)sanitize_title($text));
				}
				if ($id === '') {
					$id = 'section-' . ($idx + 1);
				}

				$base = $id;
				$suffix = 2;
				while (isset($used[$id])) {
					$id = $base . '-' . $suffix;
					$suffix++;
				}
				$used[$id] = true;

				$idx++;

				$prefix = '';
				if ($legacy_id !== '' && $legacy_id !== $id && !isset($used[$legacy_id])) {
					$used[$legacy_id] = true;
					$prefix = '<a id="' . esc_attr($legacy_id) . '" class="cr-anchor-alias" aria-hidden="true"></a>';
				}

				$attrs_out = $attrs;
				if ($attrs_out !== '' && $attrs_out[0] !== ' ') {
					$attrs_out = ' ' . ltrim($attrs_out);
				}
				$attrs_out .= ' id="' . esc_attr($id) . '"';

				return $prefix . '<' . $tag . $attrs_out . '>' . $inner . '</' . $tag . '>';
			},
			$html
		);
	}
}
