<?php

declare(strict_types=1);

namespace ContentReady\Frontend\ModuleRenderer;

use ContentReady\Admin\Settings;

final class TocRenderer
{
	public static function render(string $label, array $data, string $anchor): string
	{
		$items_in = is_array($data['items'] ?? null) ? $data['items'] : [];
		if ($items_in === []) {
			return '';
		}

		$options = Settings::get_options();
		$modules_cfg = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$cfg = is_array($modules_cfg['toc'] ?? null) ? $modules_cfg['toc'] : [];
		$rules = is_array($cfg['toc'] ?? null) ? $cfg['toc'] : [];

		$open = !empty($rules['default_open']);
		$list_style = sanitize_key((string)($rules['list_style'] ?? 'unordered'));
		if (!in_array($list_style, ['unordered', 'ordered'], true)) {
			$list_style = 'unordered';
		}
		$max_level = (int)($rules['max_level'] ?? 4);
		$max_level = max(2, min(5, $max_level));

		$items = [];
		foreach ($items_in as $it) {
			if (!is_array($it)) {
				continue;
			}
			$id = sanitize_text_field((string)($it['id'] ?? ''));
			$text = sanitize_text_field((string)($it['text'] ?? ''));
			$level = (int)($it['level'] ?? 2);
			$level = max(2, min(5, $level));
			if ($level > $max_level) {
				continue;
			}
			if ($id === '' || $text === '') {
				continue;
			}
			$items[] = ['id' => $id, 'text' => $text, 'level' => $level];
			if (count($items) >= 200) {
				break;
			}
		}

		// 少于 2 个标题时不展示目录（避免噪音）。
		if (count($items) < 2) {
			return '';
		}

		$tag = $list_style === 'ordered' ? 'ol' : 'ul';
		$list_class = 'cr-toc__list cr-toc__list--' . esc_attr($tag);
		$sub_class = 'cr-toc__sublist cr-toc__sublist--' . esc_attr($tag);

		$body = '<' . $tag . ' class="' . $list_class . '">';
		$current_level = 2;
		$first = true;

		foreach ($items as $it) {
			$level = (int)$it['level'];
			// 防止跳级导致非法嵌套：最多只下降/上升 1 级。
			if ($first) {
				if ($level > 2) {
					$level = 2;
				}
				$first = false;
			} elseif ($level > $current_level + 1) {
				$level = $current_level + 1;
			}

			if ($level === $current_level) {
				$body .= '</li>';
			} elseif ($level > $current_level) {
				$body .= '<' . $tag . ' class="' . $sub_class . '">';
				$current_level++;
			} else {
				$body .= '</li>';
				while ($current_level > $level) {
					$body .= '</' . $tag . '></li>';
					$current_level--;
				}
			}

			$body .= '<li class="cr-toc__item cr-toc__item--l' . esc_attr((string)$level) . '"><a href="#' . esc_attr((string)$it['id']) . '">' . esc_html((string)$it['text']) . '</a>';
		}

		$body .= '</li>';
		while ($current_level > 2) {
			$body .= '</' . $tag . '></li>';
			$current_level--;
		}
		$body .= '</' . $tag . '>';

		$open_attr = $open ? ' open' : '';
		$attr = ' class="cr-module cr-module--toc"';
		if ($anchor !== '') {
			$attr .= ' id="' . esc_attr($anchor) . '"';
		}

		return '<section' . $attr . '>' .
			'<details class="cr-toc"' . $open_attr . '>' .
			'<summary class="cr-toc__summary">' . esc_html($label) . '</summary>' .
			'<div class="cr-toc__body">' . $body . '</div>' .
			'</details>' .
			'</section>';
	}
}

