<?php

declare(strict_types=1);

namespace ContentReady\Frontend\ModuleRenderer;

final class TableRenderer
{
	public static function render(string $label, array $data, string $anchor): string
	{
		$caption = sanitize_text_field((string)($data['caption'] ?? ''));
		$notes = sanitize_text_field((string)($data['notes'] ?? ''));
		$cols = is_array($data['columns'] ?? null) ? $data['columns'] : [];
		$rows = is_array($data['rows'] ?? null) ? $data['rows'] : [];

		$cols = array_values(array_filter(array_map('sanitize_text_field', array_map('strval', $cols)), static fn($s) => $s !== ''));
		$clean_rows = [];
		foreach ($rows as $row) {
			if (!is_array($row)) {
				continue;
			}
			$r = array_values(array_map('sanitize_text_field', array_map('strval', $row)));
			$r = array_map(static fn($s) => trim((string)$s), $r);
			if ($r !== []) {
				$clean_rows[] = $r;
			}
		}
		$rows = $clean_rows;

		if ($cols === [] || $rows === []) {
			$msg = $notes !== '' ? $notes : '本文信息密度不足以生成表格表单';
			return Wrapper::wrap('table', $anchor, $label, '<div class="cr-module__body cr-module__muted">' . esc_html($msg) . '</div>');
		}

		$out = '<div class="cr-module__body">';
		$out .= '<div class="cr-table-wrap"><table class="cr-table">';
		if ($caption !== '') {
			$out .= '<caption>' . esc_html($caption) . '</caption>';
		}
		$out .= '<thead><tr>';
		foreach ($cols as $c) {
			$out .= '<th scope="col">' . esc_html((string)$c) . '</th>';
		}
		$out .= '</tr></thead><tbody>';
		foreach ($rows as $row) {
			if (!is_array($row)) {
				continue;
			}
			$out .= '<tr>';
			foreach ($cols as $idx => $_c) {
				$cell = isset($row[$idx]) ? (string)$row[$idx] : '';
				$out .= '<td>' . esc_html($cell) . '</td>';
			}
			$out .= '</tr>';
		}
		$out .= '</tbody></table></div>';
		if ($notes !== '') {
			$out .= '<p class="cr-table__notes">' . esc_html($notes) . '</p>';
		}
		$out .= '</div>';

		return Wrapper::wrap('table', $anchor, $label, $out);
	}
}

