<?php

declare(strict_types=1);

namespace ContentReady\Frontend;

use ContentReady\Admin\Settings;
use ContentReady\Frontend\ModuleRenderer\LastUpdatedRenderer;
use ContentReady\Frontend\ModuleRenderer\TableRenderer;
use ContentReady\Frontend\ModuleRenderer\TocRenderer;
use ContentReady\Frontend\ModuleRenderer\Wrapper;

final class ModuleRenderer
{
	public static function render(string $module_id, string $label, array $data, string $anchor): string
	{
		$label = $label !== '' ? $label : $module_id;

		switch ($module_id) {
			case 'summary':
				$text = sanitize_text_field((string)($data['text'] ?? ''));
				if ($text === '') {
					return '';
				}
				return Wrapper::wrap($module_id, $anchor, $label, '<div class="cr-module__body">' . esc_html($text) . '</div>');

			case 'verdict':
				$text = (string)($data['text'] ?? '');
				if ($text === '') {
					return '';
				}
				return Wrapper::wrap($module_id, $anchor, $label, '<div class="cr-module__body">' . $text . '</div>');

			case 'toc':
				return TocRenderer::render($label, $data, $anchor);

			case 'tldr':
			case 'limitations':
				$items = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items === []) {
					return '';
				}
				$list = '<ul class="cr-module__list">';
				foreach ($items as $item) {
					$list .= '<li>' . (string)$item . '</li>';
				}
				$list .= '</ul>';
				return Wrapper::wrap($module_id, $anchor, $label, $list);

			case 'faq':
				$items = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items === []) {
					return '';
				}
				$out = '<div class="cr-faq">';
				foreach ($items as $item) {
					if (!is_array($item)) {
						continue;
					}
					$q = isset($item['q']) ? esc_html((string)$item['q']) : '';
					$a = isset($item['a']) ? (string)$item['a'] : '';
					if ($q === '' || $a === '') {
						continue;
					}
					$out .= '<div class="cr-faq__item">';
					$out .= '<div class="cr-faq__q"><strong>Q：</strong>' . $q . '</div>';
					$out .= '<div class="cr-faq__a"><strong>A：</strong>' . $a . '</div>';
					$out .= '</div>';
				}
				$out .= '</div>';
				return Wrapper::wrap($module_id, $anchor, $label, $out);

			case 'references':
				$items = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items === []) {
					return '';
				}
				$out = '<ul class="cr-module__list">';
				foreach ($items as $item) {
					if (!is_array($item)) {
						continue;
					}
					$title = esc_html((string)($item['title'] ?? ''));
					$url = esc_url((string)($item['url'] ?? ''));
					if ($url === '') {
						continue;
					}
					$text = $title !== '' ? $title : $url;
					$out .= '<li><a href="' . $url . '" rel="nofollow noopener" target="_blank">' . $text . '</a></li>';
				}
				$out .= '</ul>';
				return Wrapper::wrap($module_id, $anchor, $label, $out);

			case 'last_updated':
				return LastUpdatedRenderer::render($label, $data, $anchor);

			case 'table':
				return TableRenderer::render($label, $data, $anchor);

			case 'slug_suggestions':
				// 仅编辑器辅助功能：不在前端渲染。
				return '';

			default:
				return '';
		}
	}
}
