<?php

declare(strict_types=1);

namespace ContentReady\Compat;

use ContentReady\Admin\Settings;
use ContentReady\Meta\MetaStore;

final class JsonLdDetector
{
	public static function register(): void
	{
		add_action('save_post', [self::class, 'schedule_check'], 20, 3);
		add_action('cr_meta_updated', [self::class, 'schedule_check_on_meta_updated'], 20, 2);
		add_action('wp', [self::class, 'schedule_check_on_front'], 20);
		add_action('cr_check_jsonld_conflict', [self::class, 'run_check'], 10, 1);
	}

	public static function schedule_check(int $post_id, \WP_Post $post, bool $update): void
	{
		if (!self::should_run_auto_check()) {
			return;
		}
		self::maybe_schedule_check($post_id, $post);
	}

	public static function schedule_check_on_meta_updated(int $post_id, array $meta): void
	{
		if (doing_action('cr_check_jsonld_conflict')) {
			return;
		}
		if (!self::should_run_auto_check()) {
			return;
		}

		$post = get_post($post_id);
		if (!$post) {
			return;
		}

		self::maybe_schedule_check($post_id, $post);
	}

	public static function schedule_check_on_front(): void
	{
		if (is_admin() || !is_singular()) {
			return;
		}

		$ua = isset($_SERVER['HTTP_USER_AGENT']) ? (string)$_SERVER['HTTP_USER_AGENT'] : '';
		if ($ua !== '' && stripos($ua, 'ContentReady/') !== false) {
			// 避免自检任务的 wp_remote_get 抓取页面时触发递归自检。
			return;
		}

		if (!self::should_run_auto_check()) {
			return;
		}

		$post_id = (int)get_queried_object_id();
		if ($post_id <= 0) {
			return;
		}

		$post = get_post($post_id);
		if (!$post) {
			return;
		}

		$meta = MetaStore::get($post_id);
		$checked_at = (int)($meta['compat']['checked_at'] ?? 0);
		if ($checked_at > 0) {
			return;
		}

		// 若站点禁用 WP-Cron，则改为同步自检（仅首次）；否则调度异步自检。
		if (defined('DISABLE_WP_CRON') && DISABLE_WP_CRON) {
			self::run_check($post_id);
			return;
		}

		self::maybe_schedule_check($post_id, $post);
	}

	private static function should_run_auto_check(): bool
	{
		$options = Settings::get_options();
		$enabled = !empty($options['structured_output']['enabled']);
		$mode = (string)($options['structured_output']['mode'] ?? 'auto'); // auto | force | off

		// 兼容性自检仅用于“智能兼容（auto）”下的冲突探测与自动暂停输出。
		return $enabled && $mode === 'auto';
	}

	private static function maybe_schedule_check(int $post_id, \WP_Post $post): void
	{
		if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
			return;
		}
		if ($post->post_status !== 'publish') {
			return;
		}

		if (wp_next_scheduled('cr_check_jsonld_conflict', [$post_id])) {
			return;
		}

		wp_schedule_single_event(time() + 30, 'cr_check_jsonld_conflict', [$post_id]);
	}

	public static function run_check(int $post_id): void
	{
		$post = get_post($post_id);
		if (!$post || $post->post_status !== 'publish') {
			return;
		}

		$url = get_permalink($post);
		if (!$url) {
			return;
		}

		$response = wp_remote_get($url, [
			'timeout' => 10,
			'redirection' => 3,
			'user-agent' => 'ContentReady/' . (defined('CR_VERSION') ? CR_VERSION : '0'),
		]);

		$meta = MetaStore::get($post_id);
		$meta['compat']['checked_at'] = time();

		if (is_wp_error($response)) {
			$meta['compat']['jsonld_conflict'] = false;
			$meta['compat']['last_error'] = $response->get_error_message();
			MetaStore::update($post_id, $meta);
			return;
		}

		$body = (string)wp_remote_retrieve_body($response);
		if ($body === '') {
			$meta['compat']['jsonld_conflict'] = false;
			$meta['compat']['last_error'] = 'empty_response_body';
			MetaStore::update($post_id, $meta);
			return;
		}

		$head = self::extract_head($body);
		$tail = self::extract_body_tail($body, 150000);

		$has_other = self::has_other_jsonld($head) || self::has_other_jsonld($tail);
		$meta['compat']['jsonld_conflict'] = $has_other;
		$meta['compat']['last_error'] = '';

		MetaStore::update($post_id, $meta);
	}

	private static function extract_head(string $html): string
	{
		if (preg_match('~<head\\b[^>]*>(.*?)</head>~is', $html, $m) === 1) {
			return (string)$m[1];
		}
		return '';
	}

	private static function extract_body_tail(string $html, int $max_len): string
	{
		$pos = strripos($html, '</body>');
		$end = $pos !== false ? $pos : strlen($html);
		$start = max(0, $end - $max_len);
		return substr($html, $start, $end - $start);
	}

	private static function has_other_jsonld(string $fragment): bool
	{
		if ($fragment === '') {
			return false;
		}

		if (preg_match_all('~<script\\b[^>]*type\\s*=\\s*[\"\\\']application/ld\\+json[\"\\\'][^>]*>~i', $fragment, $m) !== false) {
			foreach ($m[0] as $tag) {
				if (stripos((string)$tag, 'data-cr-jsonld') === false) {
					return true;
				}
			}
		}

		return false;
	}
}
