<?php

declare(strict_types=1);

namespace ContentReady\Automation;

use ContentReady\Admin\Settings;
use ContentReady\Licensing\Capabilities;
use ContentReady\Licensing\AutomationGate;
use ContentReady\Licensing\DailyQuota;

final class Backfill
{
	private const STATE_OPTION_KEY = 'cr_backfill_state_v1';
	private const DAILY_HOOK = 'cr_backfill_daily';
	private const TICK_HOOK = 'cr_backfill_tick';

	public static function register(): void
	{
		add_action(self::DAILY_HOOK, [self::class, 'run_daily'], 10);
		add_action(self::TICK_HOOK, [self::class, 'run_tick'], 10);

		add_action('update_option_' . Settings::OPTION_KEY, [self::class, 'on_options_updated'], 10, 2);
	}

	public static function on_options_updated($old, $new): void
	{
		$options = Settings::get_options();
		self::sync_schedule($options);
	}

	public static function sync_schedule(array $options): void
	{
		$is_pro = Capabilities::feature_enabled(Capabilities::FEATURE_BACKFILL, $options);
		$gen = is_array($options['generation'] ?? null) ? $options['generation'] : [];
		$enabled = $is_pro && !empty($gen['backfill_enabled']);
		$mode = sanitize_key((string)($gen['backfill_mode'] ?? 'once')); // once | daily
		if (!in_array($mode, ['once', 'daily'], true)) {
			$mode = 'once';
		}

		// 关闭时：清理调度
		if (!$enabled) {
			self::clear_schedules();
			return;
		}

		// 若之前已完成，重新启用时从头开始（用户显式开启即视为重新跑一遍）。
		$state = get_option(self::STATE_OPTION_KEY, []);
		$state = is_array($state) ? $state : [];
		if (!empty($state['done'])) {
			$state['done'] = false;
			$state['offset'] = 0;
			unset($state['completed_at']);
			update_option(self::STATE_OPTION_KEY, $state, false);
		}

		if ($mode === 'daily') {
			self::schedule_daily($options);
			self::clear_tick();
			return;
		}

		// once：立即开始，直到完成
		self::clear_daily();
		self::ensure_tick_scheduled(15);
	}

	private static function schedule_daily(array $options): void
	{
		$hour = (int)($options['generation']['backfill_hour'] ?? 3);
		$hour = max(0, min(23, $hour));

		$next = self::next_timestamp_at_hour($hour);
		$current = wp_next_scheduled(self::DAILY_HOOK);
		if ($current) {
			// 若下一次执行时间相差较大，重建调度
			if (abs((int)$current - (int)$next) > 60) {
				wp_unschedule_event((int)$current, self::DAILY_HOOK);
				wp_schedule_event((int)$next, 'daily', self::DAILY_HOOK);
			}
			return;
		}

		wp_schedule_event((int)$next, 'daily', self::DAILY_HOOK);
	}

	private static function next_timestamp_at_hour(int $hour): int
	{
		if (function_exists('wp_timezone')) {
			$tz = wp_timezone();
		} else {
			$tz = new \DateTimeZone('UTC');
		}

		$now = new \DateTime('now', $tz);
		$run = (clone $now)->setTime($hour, 0, 0);
		if ($run <= $now) {
			$run = $run->modify('+1 day');
		}
		return (int)$run->format('U');
	}

	public static function run_daily(): void
	{
		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_BACKFILL, $options)) {
			return;
		}
		$gen = is_array($options['generation'] ?? null) ? $options['generation'] : [];
		if (empty($gen['backfill_enabled'])) {
			return;
		}
		if (sanitize_key((string)($gen['backfill_mode'] ?? 'once')) !== 'daily') {
			return;
		}

		self::run_tick();
	}

	public static function run_tick(): void
	{
		$raw = Settings::get_raw_options();
		$gen0 = is_array($raw['generation'] ?? null) ? $raw['generation'] : [];
		if (empty($gen0['backfill_enabled'])) {
			return;
		}

                $gate = AutomationGate::ensure_ticket('automation_backfill');
                if (empty($gate['allowed'])) {
                        $msg = sanitize_text_field((string)($gate['message'] ?? ''));
                        if ($msg !== '') {
                                AutomationGate::record_notice('automation_backfill', (string)($gate['reason'] ?? 'blocked'), $msg);
                        }
                        return;
                }

		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_BACKFILL, $options)) {
			return;
		}
		$gen = is_array($options['generation'] ?? null) ? $options['generation'] : [];
		if (empty($gen['backfill_enabled'])) {
			return;
		}

		$mode = sanitize_key((string)($gen['backfill_mode'] ?? 'once')); // once | daily
		if (!in_array($mode, ['once', 'daily'], true)) {
			$mode = 'once';
		}

		$limit = 10;
		if ($mode === 'daily') {
			$limit = (int)($gen['backfill_daily_limit'] ?? 10);
			$limit = max(1, min(200, $limit));
		}

		$state = get_option(self::STATE_OPTION_KEY, []);
		$state = is_array($state) ? $state : [];

                $offset = (int)($state['offset'] ?? 0);
                if (!empty($state['done'])) {
                        return;
                }

		$ids = get_posts([
			'post_type'      => ['post', 'page'],
			'post_status'    => 'publish',
			'orderby'        => 'ID',
			'order'          => 'ASC',
			'posts_per_page' => $limit,
			'offset'         => $offset,
			'fields'         => 'ids',
		]);

		if (!is_array($ids) || $ids === []) {
			$state['done'] = true;
			$state['completed_at'] = time();
			update_option(self::STATE_OPTION_KEY, $state, false);
			self::clear_tick();
			return;
		}

		$processed = 0;
		foreach ($ids as $id) {
			$check = DailyQuota::can_generate_post((int)$id, $options);
			if (empty($check['ok'])) {
				$msg = 'Content Ready：老文章回填未执行（' . (string)($check['message'] ?? '已达今日配额。') . '）';
				DailyQuota::record_automation_block('automation_backfill', $msg);
				break;
			}

			GeoRunner::run_for_post((int)$id, true, true);
			$processed++;
		}

		$state['offset'] = $offset + $processed;
		update_option(self::STATE_OPTION_KEY, $state, false);

		// once：继续跑直到完成；daily：当天到此为止
		if ($mode === 'once') {
			self::ensure_tick_scheduled(60);
		}
	}

	private static function ensure_tick_scheduled(int $delay_seconds): void
	{
		if (wp_next_scheduled(self::TICK_HOOK)) {
			return;
		}
		wp_schedule_single_event(time() + max(5, $delay_seconds), self::TICK_HOOK);
	}

	private static function clear_tick(): void
	{
		$ts = wp_next_scheduled(self::TICK_HOOK);
		if ($ts) {
			wp_unschedule_event((int)$ts, self::TICK_HOOK);
		}
	}

	private static function clear_daily(): void
	{
		$ts = wp_next_scheduled(self::DAILY_HOOK);
		if ($ts) {
			wp_unschedule_event((int)$ts, self::DAILY_HOOK);
		}
	}

	private static function clear_schedules(): void
	{
		self::clear_daily();
		self::clear_tick();
	}
}
