<?php

declare(strict_types=1);

namespace ContentReady\Automation;

use ContentReady\Admin\Settings;
use ContentReady\Licensing\Capabilities;
use ContentReady\Licensing\AutomationGate;
use ContentReady\Licensing\DailyQuota;

final class AutoGeo
{
        public static function register(): void
        {
                add_action('save_post', [self::class, 'maybe_schedule_on_save_post'], 55, 3);
                add_action('cr_auto_geo_generate', [self::class, 'run_generate'], 10, 1);
	}

	public static function maybe_schedule_on_save_post(int $post_id, \WP_Post $post, bool $update): void
	{
		if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
			return;
		}
		if ($post->post_status !== 'publish') {
			return;
		}
		if (!current_user_can('edit_post', $post_id)) {
			return;
		}

		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_AUTO_GEO_ON_PUBLISH, $options)) {
			return;
		}
                if (empty($options['generation']['auto_geo_on_publish'])) {
                        return;
                }

                if (wp_next_scheduled('cr_auto_geo_generate', [$post_id])) {
                        return;
                }

		// 异步调度：不阻塞发布/更新。
		wp_schedule_single_event(time() + 10, 'cr_auto_geo_generate', [$post_id]);
	}

	public static function run_generate(int $post_id): void
	{
		$raw = Settings::get_raw_options();
		$gen = is_array($raw['generation'] ?? null) ? $raw['generation'] : [];
                if (empty($gen['auto_geo_on_publish'])) {
                        return;
                }

                $gate = AutomationGate::ensure_ticket('automation_auto_geo');
                if (empty($gate['allowed'])) {
                        $msg = sanitize_text_field((string)($gate['message'] ?? ''));
                        if ($msg !== '') {
                                AutomationGate::record_notice('automation_auto_geo', (string)($gate['reason'] ?? 'blocked'), $msg);
                        }
                        return;
                }

                $options = Settings::get_options();
                if (!Capabilities::feature_enabled(Capabilities::FEATURE_AUTO_GEO_ON_PUBLISH, $options)) {
                        return;
                }
		if (empty($options['generation']['auto_geo_on_publish'])) {
			return;
		}

                $check = DailyQuota::can_generate_post($post_id, $options);     
                if (empty($check['ok'])) {
                        $msg = 'Content Ready：发布时自动生成 GEO 未执行（' . (string)($check['message'] ?? '已达今日配额。') . '不影响文章发布）';
                        DailyQuota::record_automation_block('automation_auto_geo', $msg);
                        return;
                }

		GeoRunner::run_for_post($post_id, true, true);
	}
}
