<?php

declare(strict_types=1);

namespace ContentReady\Admin\Tabs;

use ContentReady\Admin\AiProviderLinks;
use ContentReady\Admin\SettingsForm;
use ContentReady\Admin\Ui\Badges;
use ContentReady\Licensing\Capabilities;

final class AiPrivacyTab
{
	public static function render(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);
		$ai = is_array($options['ai'] ?? null) ? $options['ai'] : [];

		echo '<div class="notice notice-info inline" style="margin-top:16px;"><p>';
		echo esc_html__('Content Ready 不提供 AI Key：AI 生成基于你自有的接口配置。', 'content-ready');
		echo '</p></div>';

		echo '<h2 style="margin-top:16px;">' . esc_html__('AI 与隐私', 'content-ready') . '</h2>';
		echo '<p class="description">把下面 3 项填好就能开始用了：接口地址（Base URL）/ Key / 模型。</p>';

		$provider_links = AiProviderLinks::links();
		$provider_links = is_array($provider_links) ? $provider_links : [];
		$provider_links = array_values(array_filter($provider_links, static function ($item): bool {
			if (!is_array($item)) {
				return false;
			}
			$label = trim((string)($item['label'] ?? ''));
			$url = trim((string)($item['url'] ?? ''));
			return $label !== '' && $url !== '';
		}));
		$provider_links = array_slice($provider_links, 0, 3);
		if ($provider_links !== []) {
			echo '<div class="notice notice-info inline" style="margin-top:12px;"><p style="margin:0;">还没有 AI 接口？你可以从下面任选其一（新标签打开）：</p>';
			echo '<ul style="margin:8px 0 0;padding-left:18px;">';
			foreach ($provider_links as $item) {
				$label = sanitize_text_field((string)($item['label'] ?? ''));
				$url = esc_url_raw((string)($item['url'] ?? ''));
				if ($label === '' || $url === '') {
					continue;
				}
				echo '<li><a href="' . esc_url($url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($label) . '</a></li>';
			}
			echo '</ul>';
			echo '</div>';
		}
		SettingsForm::open('ai_privacy');

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row"><label for="cr_ai_base_url">AI 接口地址（API Base）</label></th><td>';
		echo '<input class="regular-text" id="cr_ai_base_url" name="ai_base_url" value="' . esc_attr((string)($ai['base_url'] ?? '')) . '" placeholder="https://api.example.com/v1" />';
		echo '<p class="description">支持 OpenAI 兼容接口；请填写你购买/自建的网关地址。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_ai_api_key">AI Key</label></th><td>';
		echo '<input class="regular-text" id="cr_ai_api_key" name="ai_api_key" value="' . esc_attr((string)($ai['api_key'] ?? '')) . '" autocomplete="off" />';
		echo '<p class="description">用于调用你的 AI 接口。注意：这会影响你的 Token 消耗与费用。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_ai_model">模型</label></th><td>';
		echo '<input class="regular-text" id="cr_ai_model" name="ai_model" value="' . esc_attr((string)($ai['model'] ?? '')) . '" placeholder="例如：gpt-4o-mini / deepseek-chat / llama-3.1-70b" />';
		echo '<p class="description">模型名称不做限定：请填写你的 AI 服务商实际支持的模型标识（不同服务商/渠道命名可能不同）。</p>';
		echo '</td></tr>';

		$timeout_sec = (int)($ai['request_timeout_sec'] ?? 90);
		$timeout_sec = max(20, min(300, $timeout_sec));
		echo '<tr><th scope="row">超时控制</th><td>';
		echo '<label for="cr_ai_request_timeout_sec">单次超时（秒）</label> ';
		echo '<input type="number" min="20" max="300" style="width:100px" id="cr_ai_request_timeout_sec" name="ai_request_timeout_sec" value="' . esc_attr((string)$timeout_sec) . '" /> ';
		echo '<p class="description">建议：重模型可设为 120-180 秒，常规模型可设为 60-90 秒。</p>';
		echo '</td></tr>';

		$probe_last_at = (int)($ai['probe_last_at'] ?? 0);
		$probe_last_ok = !empty($ai['probe_last_ok']);
		$probe_last_message = sanitize_text_field((string)($ai['probe_last_message'] ?? ''));
		$probe_status = $probe_last_at > 0 ? ($probe_last_ok ? '成功' : '失败') : '未测通';
		$probe_time_text = $probe_last_at > 0 ? wp_date('Y-m-d H:i:s', $probe_last_at) : '—';
		echo '<tr><th scope="row">最近一次测通结果</th><td>';
		echo '<p class="description" style="margin:0;">状态：' . esc_html($probe_status) . '；时间：' . esc_html($probe_time_text) . '</p>';
		if ($probe_last_message !== '') {
			echo '<p class="description" style="margin-top:4px;">详情：' . esc_html($probe_last_message) . '</p>';
		}
		echo '</td></tr>';

		$probe_nonce = wp_create_nonce('cr_probe_ai');
		echo '<tr><th scope="row">AI 测通（真实调用）</th><td>';
		echo '<button type="button" class="button button-secondary" id="cr-ai-probe-btn" data-cr-nonce="' . esc_attr($probe_nonce) . '">测通</button>';
		echo ' <span id="cr-ai-probe-msg" class="description" style="margin-left:8px;"></span>';
		echo '<p class="description">使用当前输入的 Base URL / Key / 模型发起一次真实调用，并要求 AI 回显固定文本（会消耗少量 Token）。</p>';
		echo '</td></tr>';

		$ai_mode = sanitize_key((string)($ai['mode'] ?? 'standard'));
		if (!in_array($ai_mode, ['quick', 'standard', 'high'], true)) {
			$ai_mode = 'standard';
		}
		echo '<tr><th scope="row"><label for="cr_ai_mode">生成模式</label></th><td>';
		echo '<select id="cr_ai_mode" name="ai_mode">';
		foreach (['quick' => '快速（更简洁）', 'standard' => '标准（默认）', 'high' => '高质量（更全面）'] as $k => $label) {
			echo '<option value="' . esc_attr($k) . '"' . selected($ai_mode, $k, false) . '>' . esc_html($label) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">影响提示词对“输出长度/条数”的倾向：快速更省 Token、更快；高质量更全面、通常更耗时。参考：摘要约 80–260 字；TL;DR/适用范围通常 3–7 条；FAQ 通常 3–8 条（实际取决于文章长度、启用模块与模型能力）。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">隐私：发送范围 ' . Badges::pro('解锁“标题 + 短摘要”') . '</th><td>';
		$scope = (string)($options['privacy']['send_scope'] ?? 'full');
		echo '<select name="privacy_send_scope">';
		$choices = $is_pro
			? ['full' => '全文（推荐）', 'title_summary' => '标题 + 摘要', 'selected_modules' => '标题 + 短摘要（Pro）']
			: ['full' => '全文（推荐）', 'title_summary' => '标题 + 摘要'];
		foreach ($choices as $k => $label) {
			echo '<option value="' . esc_attr($k) . '"' . selected($scope, $k, false) . '>' . esc_html($label) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">用于控制发送给 AI 的上下文范围。默认推荐“全文”：更贴合文章语境、效果更稳。范围越小越省 Token/更少暴露内容，但生成效果可能更不稳定；如果你的目标主要是省 Token，优先考虑减少启用模块数量或使用「轻量精选模块调度」。</p>';
		if (!$is_pro) {
			echo '<p class="description">Free：暂不支持“标题 + 短摘要”。</p>';
		}
		echo '</td></tr>';

		echo '</tbody></table>';
		SettingsForm::close();
	}
}
