<?php

declare(strict_types=1);

namespace ContentReady\Admin\SettingsSections;

use ContentReady\SiteEntrances\WpSitemapDiagnostics;

final class SiteEntrancesSection
{
	public static function render(array $options): void
	{
		$se = is_array($options['site_entrances'] ?? null) ? $options['site_entrances'] : [];

		$llms_enabled = !empty($se['llms_enabled']);
		$wp_sitemap_control_enabled = !empty($se['wp_sitemap_control_enabled']);
		$wp_posts = array_key_exists('wp_sitemap_include_posts', $se) ? !empty($se['wp_sitemap_include_posts']) : true;
		$wp_pages = array_key_exists('wp_sitemap_include_pages', $se) ? !empty($se['wp_sitemap_include_pages']) : true;
		$wp_tags = !empty($se['wp_sitemap_include_tags']);
		$wp_users = !empty($se['wp_sitemap_include_users']);
		$recommended = (string)($se['recommended_urls'] ?? '');

		$llms_url = home_url('/llms.txt');
		$wp_sitemap_url = home_url('/wp-sitemap.xml');

		echo '<div class="notice notice-info inline" style="margin:12px 0 0;"><p style="margin:0;">';
		echo '预览（新标签打开）：';
		echo '<a href="' . esc_url($llms_url) . '" target="_blank" rel="noopener noreferrer">/llms.txt</a>';
		echo ' ｜ ';
		echo '<a href="' . esc_url($wp_sitemap_url) . '" target="_blank" rel="noopener noreferrer">/wp-sitemap.xml</a>';
		echo '</p></div>';

		$force_diag = !empty($_GET['cr_wp_sitemap_diag']) && current_user_can('manage_options');
		if ($force_diag) {
			WpSitemapDiagnostics::clear_cache();
		}
		$diag = WpSitemapDiagnostics::get($force_diag);
		$diag_checked_at = (int)($diag['checked_at'] ?? 0);
		$diag_enabled = $diag['enabled'] ?? null;
		$diag_http = is_array($diag['http'] ?? null) ? $diag['http'] : [];
		$diag_http_ok = !empty($diag_http['ok']);
		$diag_http_status = (int)($diag_http['status'] ?? 0);
		$diag_http_ct = (string)($diag_http['content_type'] ?? '');
		$diag_http_looks = array_key_exists('looks_like_sitemap', $diag_http) ? $diag_http['looks_like_sitemap'] : null;
		$diag_http_err = (string)($diag_http['error_message'] ?? '');

		$refresh_url = add_query_arg(
			[
				'page' => 'content-ready-settings',
				'tab' => 'site_entrances',
				'cr_wp_sitemap_diag' => '1',
			],
			admin_url('admin.php')
		);

		if ($diag_enabled === false) {
			echo '<div class="notice notice-warning inline" style="margin:12px 0 0;"><p style="margin:0;">';
			echo '<strong>检测结果：</strong>WordPress 自带 Sitemap 已被禁用（wp-sitemap.xml）。';
			echo ' 通常是 SEO 插件接管/禁用了 WP sitemap，或站点策略禁用了 sitemap。';
			echo ' 若你使用 RankMath/Yoast，请到对应插件里管理 sitemap；本页开关可能不会生效。';
			if ($diag_checked_at > 0) {
				echo '（检测时间：' . esc_html(wp_date('Y-m-d H:i', $diag_checked_at)) . '）';
			}
			echo ' <a href="' . esc_url($refresh_url) . '">重新检测</a>';
			echo '</p></div>';
		} elseif ($diag_http_ok && $diag_http_status === 200 && $diag_http_looks === true) {
			echo '<div class="notice notice-success inline" style="margin:12px 0 0;"><p style="margin:0;">';
			echo '<strong>检测结果：</strong>wp-sitemap.xml 可访问（HTTP 200）。';
			if ($diag_checked_at > 0) {
				echo '（检测时间：' . esc_html(wp_date('Y-m-d H:i', $diag_checked_at)) . '）';
			}
			echo ' <a href="' . esc_url($refresh_url) . '">重新检测</a>';
			echo '</p></div>';
		} elseif ($diag_http_ok && $diag_http_status > 0) {
			echo '<div class="notice notice-warning inline" style="margin:12px 0 0;"><p style="margin:0;">';
			echo '<strong>检测结果：</strong>wp-sitemap.xml 访问异常（HTTP ' . esc_html((string)$diag_http_status) . '）。';
			if ($diag_http_ct !== '') {
				echo ' Content-Type：' . esc_html($diag_http_ct) . '。';
			}
			if ($diag_http_status === 404) {
				echo ' 可能原因：固定链接规则/重写异常，或 SEO 插件禁用/替换了 WP sitemap。';
			} elseif ($diag_http_looks === false) {
				echo ' 返回内容不像 sitemap（可能被重定向到 HTML 页面）。';
			}
			if ($diag_checked_at > 0) {
				echo '（检测时间：' . esc_html(wp_date('Y-m-d H:i', $diag_checked_at)) . '）';
			}
			echo ' <a href="' . esc_url($refresh_url) . '">重新检测</a>';
			echo '</p></div>';
		} elseif ($diag_http_err !== '') {
			echo '<div class="notice notice-warning inline" style="margin:12px 0 0;"><p style="margin:0;">';
			echo '<strong>检测结果：</strong>无法检测 wp-sitemap.xml（回环请求失败）。';
			echo ' 错误：' . esc_html($diag_http_err) . '。';
			echo ' 常见原因：安全/防火墙插件阻断站点回环请求（loopback），或 HTTPS/证书/DNS 配置异常。';
			echo '（不一定代表外网不可访问，可在浏览器直接打开 wp-sitemap.xml 再确认。）';
			if ($diag_checked_at > 0) {
				echo '（检测时间：' . esc_html(wp_date('Y-m-d H:i', $diag_checked_at)) . '）';
			}
			echo ' <a href="' . esc_url($refresh_url) . '">重新检测</a>';
			echo '</p></div>';
		}

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row">输出 /llms.txt</th><td>';
		echo '<label><input type="checkbox" name="site_entrances_llms_enabled" value="1"' . checked($llms_enabled, true, false) . ' /> 启用</label>';
		echo '<p class="description">输出 LLM 友好的 Markdown 入口文件（llms.txt）。如你的站点已使用 RankMath/Yoast 等插件生成 llms.txt，请二选一，避免重复/冲突。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">WordPress 自带 Sitemap（wp-sitemap.xml）</th><td>';
		echo '<label><input type="checkbox" name="site_entrances_wp_sitemap_control_enabled" value="1"' . checked($wp_sitemap_control_enabled, true, false) . ' /> 由本插件接管/控制（最小实践）</label>';
		echo '<p class="description">WordPress 自带了基础的 XML Sitemap（<a href="' . esc_url($wp_sitemap_url) . '" target="_blank" rel="noopener noreferrer">wp-sitemap.xml</a>）。本插件只做“最小可控”的开关管理；如需更细腻的 sitemap 规则，请使用专业 Sitemap/SEO 插件。</p>';
		echo '<p class="description">注意：如果你已启用 RankMath/Yoast 等 SEO 插件的 sitemap 功能，它可能会接管/替换 WordPress 自带 sitemap，此处开关可能不会生效（请到对应插件里管理）。</p>';
		echo '<p style="margin:8px 0 0;">收录内容：</p>';
		echo '<label style="margin-right:12px;"><input type="checkbox" name="site_entrances_wp_sitemap_include_posts" value="1"' . checked($wp_posts, true, false) . ' /> 文章</label>';
		echo '<label style="margin-right:12px;"><input type="checkbox" name="site_entrances_wp_sitemap_include_pages" value="1"' . checked($wp_pages, true, false) . ' /> 页面</label>';
		echo '<label style="margin-right:12px;"><input type="checkbox" name="site_entrances_wp_sitemap_include_tags" value="1"' . checked($wp_tags, true, false) . ' /> 标签</label>';
		echo '<label style="margin-right:12px;"><input type="checkbox" name="site_entrances_wp_sitemap_include_users" value="1"' . checked($wp_users, true, false) . ' /> 用户</label>';
		echo '<p class="description">建议：通常默认只需要「文章」「页面」进入 sitemap 即可。标签通常繁多且内容很薄，不宜放进 sitemap；单用户站点不推荐勾选「用户」。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_site_entrances_recommended">推荐页面（可选）</label></th><td>';
		echo '<textarea class="large-text code" id="cr_site_entrances_recommended" name="site_entrances_recommended_urls" rows="7" placeholder="https://example.com/ | 首页\\nhttps://example.com/about/ | 关于我们">' . esc_textarea($recommended) . '</textarea>';
		echo '<p class="description">每行 1 个：URL 后可用 <code>|</code> 填一个标题（可选）。用于 llms.txt 中的“推荐页面/关键入口”列表（更利于 AI/爬虫快速理解站点结构）。</p>';
		echo '</td></tr>';

		echo '</tbody></table>';
	}
}
