<?php

declare(strict_types=1);

namespace ContentReady\Admin\SettingsSaver;

use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Dispatch as ModulesDispatch;

final class ModulesOutputTab
{
	public static function apply(array &$options, array $post, string $prev_dispatch_mode): void
	{
		$options['frontend']['inject_enabled'] = !empty($post['frontend_inject_enabled']);
		$options['frontend']['css_enabled'] = !empty($post['frontend_css_enabled']);

		$options['structured_output']['enabled'] = !empty($post['structured_output_enabled']);

		$new_dispatch_mode = $prev_dispatch_mode;

		// 新版：模块调度策略（off/adaptive/lightweight）
		if (isset($post['generation_dispatch_mode'])) {
			$dispatch = sanitize_key((string)$post['generation_dispatch_mode']);
			if (!in_array($dispatch, ['off', 'adaptive', 'lightweight'], true)) {
				$dispatch = 'off';
			}
			$options['generation']['dispatch_mode'] = $dispatch;
			$options['generation']['adaptive_enabled'] = ($dispatch === 'adaptive');
			$new_dispatch_mode = $dispatch;
		} else {
			// 兼容旧版：自适应开关（布尔）
			$options['generation']['adaptive_enabled'] = !empty($post['generation_adaptive_enabled']);
			$options['generation']['dispatch_mode'] = $options['generation']['adaptive_enabled'] ? 'adaptive' : 'off';
			$new_dispatch_mode = $options['generation']['dispatch_mode'];
		}

		if (isset($post['structured_output_mode'])) {
			$mode = sanitize_key((string)$post['structured_output_mode']);
			if (!in_array($mode, ['auto', 'force', 'off'], true)) {
				$mode = 'auto';
			}
			$options['structured_output']['mode'] = $mode;
		}

		// 结构化输出：输出对象（Schema）与接管范围（Scope）。
		// 用 *_present 避免其他来源（旧版表单/自定义提交）意外把配置清空。
		$present = !empty($post['structured_output_present']);
		if ($present) {
			if (!is_array($options['structured_output'] ?? null)) {
				$options['structured_output'] = [];
			}

			// objects
			if (!is_array($options['structured_output']['objects'] ?? null)) {
				$options['structured_output']['objects'] = [];
			}
			$posted_objects = isset($post['structured_output_objects']) && is_array($post['structured_output_objects'])
				? $post['structured_output_objects']
				: [];
			foreach (['site_graph', 'webpage', 'article', 'faq'] as $obj_id) {
				$obj_id = sanitize_key((string)$obj_id);
				$options['structured_output']['objects'][$obj_id] = !empty($posted_objects[$obj_id]);
			}

			// scopes
			if (!is_array($options['structured_output']['scopes'] ?? null)) {
				$options['structured_output']['scopes'] = [];
			}
			$posted_scopes = isset($post['structured_output_scopes']) && is_array($post['structured_output_scopes'])
				? $post['structured_output_scopes']
				: [];
			foreach (['home', 'singular_post', 'singular_page', 'singular_cpt', 'taxonomy', 'search'] as $scope_id) {
				$scope_id = sanitize_key((string)$scope_id);
				$raw = isset($posted_scopes[$scope_id]) ? sanitize_key((string)$posted_scopes[$scope_id]) : 'off';
				if (!in_array($raw, ['inherit', 'auto', 'force', 'off'], true)) {
					$raw = 'inherit';
				}
				$options['structured_output']['scopes'][$scope_id] = $raw;
			}
		}

		$posted_modules = isset($post['modules']) && is_array($post['modules']) ? $post['modules'] : null;
		if ($posted_modules !== null) {
			$options['modules'] = ModulesConfig::sanitize_modules_config_partial($posted_modules, $options['modules'] ?? null);
		}

		// 自适应调度：切换为 adaptive 时，默认启用所有“内容模块”作为候选池（用户可再手动关闭不需要的模块）。
		$is_switch_to_adaptive = ($prev_dispatch_mode !== 'adaptive') && ($new_dispatch_mode === 'adaptive');
		if ($is_switch_to_adaptive) {
			$modules = is_array($options['modules'] ?? null) ? $options['modules'] : [];
			foreach (ModulesDispatch::adaptive_default_module_ids() as $mid) {
				$mid = sanitize_key((string)$mid);
				if (!Capabilities::is_module_available($mid, $options)) {
					continue;
				}
				$cfg = is_array($modules[$mid] ?? null) ? $modules[$mid] : [];
				$cfg['enabled'] = true;
				$modules[$mid] = $cfg;
			}
			$options['modules'] = ModulesConfig::sanitize_modules_config($modules);
		}
	}
}

