<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Admin\SettingsSaver\AdvancedTab;
use ContentReady\Admin\SettingsSaver\AiPrivacyTab;
use ContentReady\Admin\SettingsSaver\AutomationTab;
use ContentReady\Admin\SettingsSaver\BasicTab;
use ContentReady\Admin\SettingsSaver\LicenseTab;
use ContentReady\Admin\SettingsSaver\ModulesOutputTab;
use ContentReady\Admin\SettingsSaver\PageTypesTab;
use ContentReady\Admin\SettingsSaver\SiteEntrancesTab;
use ContentReady\Licensing\LicenseRuntime;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class SettingsSaver
{
	public static function handle_save(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die('Forbidden');
		}
		check_admin_referer('cr_save_settings');

		$options = wp_parse_args(Settings::get_raw_options(), Settings::default_options());
		$prev_dispatch_mode = sanitize_key((string)($options['generation']['dispatch_mode'] ?? ''));
		if (!in_array($prev_dispatch_mode, ['off', 'adaptive', 'lightweight'], true)) {
			$prev_dispatch_mode = !empty($options['generation']['adaptive_enabled']) ? 'adaptive' : 'off';
		}
		$options['modules_order'] = ModulesConfig::default_order();
		$options['modules'] = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$options['page_types'] = PageTypesConfig::normalize_page_types($options['page_types'] ?? null);
		$options['config_version'] = Settings::CONFIG_VERSION;

		$tab = isset($_POST['cr_settings_tab']) ? sanitize_key((string)$_POST['cr_settings_tab']) : 'basic';

		switch ($tab) {
			case 'site_entrances':
				SiteEntrancesTab::apply($options, $_POST);
				break;

			case 'ai_privacy':
				AiPrivacyTab::apply($options, $_POST);
				break;

			case 'modules_output':
				ModulesOutputTab::apply($options, $_POST, $prev_dispatch_mode);
				break;

			case 'page_types':
				PageTypesTab::apply($options, $_POST);
				break;

			case 'automation':
				AutomationTab::apply($options, $_POST);
				break;

			case 'license':
				LicenseTab::apply($options, $_POST);
				break;

			case 'advanced':
				AdvancedTab::apply($options, $_POST);
				break;

			case 'basic':
			default:
				BasicTab::apply($options, $_POST);
				break;
		}

		Settings::update_options($options);
		LicenseRuntime::refresh_after_settings_saved();

		$base = admin_url('admin.php');
		if ($tab === 'automation') {
			$url = add_query_arg(['page' => 'content-ready-automation', 'cr_notice' => 'cr_saved'], $base);
		} else {
			$url = add_query_arg(['page' => 'content-ready-settings', 'tab' => $tab, 'cr_notice' => 'cr_saved'], $base);
		}
		wp_safe_redirect($url);
		exit;
	}
}
