<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Admin\Tabs\BasicTab;
use ContentReady\Admin\Tabs\AdvancedTab;
use ContentReady\Admin\Tabs\AiPrivacyTab;
use ContentReady\Admin\Tabs\ImportExportTab;
use ContentReady\Admin\Tabs\LicenseTab;
use ContentReady\Admin\Tabs\ModulesOutputTab;
use ContentReady\Admin\Tabs\PageTypesTab;
use ContentReady\Admin\Tabs\SiteEntrancesTab;

final class SettingsPage
{
	/**
	 * @param string $active_tab basic|site_entrances|ai_privacy|modules_output|page_types|license|import_export|advanced
	 */
	public static function render(string $notice, array $options, string $active_tab): void
	{
		echo '<div class="wrap cr-settings">';
		echo '<h1>' . esc_html__('Content Ready｜内容就绪', 'content-ready') . '</h1>';

		self::render_notice($notice);
		self::render_quickstart($options, $active_tab);
		self::render_tabs($active_tab);
		self::render_tab_content($active_tab, $options);

		echo '</div>';
	}

	private static function tabs(): array
	{
		return [
			'basic' => '基础设置',
			'site_entrances' => '站点入口',
			'ai_privacy' => 'AI 与隐私',
			'modules_output' => '模块与输出',
			'page_types' => '页面类型规则',
			'license' => '许可证（Pro）',
			'import_export' => '导入导出',
			'advanced' => '高级/维护',
		];
	}

	private static function render_tabs(string $active_tab): void
	{
		$tabs = self::tabs();
		if (!isset($tabs[$active_tab])) {
			$active_tab = 'basic';
		}

		echo '<h2 class="nav-tab-wrapper" style="margin-top:16px;">';
		foreach ($tabs as $k => $label) {
			$url = add_query_arg(
				['page' => 'content-ready-settings', 'tab' => $k],
				admin_url('admin.php')
			);
			$cls = 'nav-tab' . ($k === $active_tab ? ' nav-tab-active' : '');
			echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
		}
		echo '</h2>';
	}

	private static function render_tab_content(string $active_tab, array $options): void
	{
		$tabs = self::tabs();
		if (!isset($tabs[$active_tab])) {
			$active_tab = 'basic';
		}

		switch ($active_tab) {
			case 'site_entrances':
				SiteEntrancesTab::render($options);
				return;
			case 'ai_privacy':
				AiPrivacyTab::render($options);
				return;
			case 'modules_output':
				ModulesOutputTab::render($options);
				return;
			case 'page_types':
				PageTypesTab::render($options);
				return;
			case 'license':
				LicenseTab::render($options);
				return;
			case 'import_export':
				ImportExportTab::render();
				return;
			case 'advanced':
				AdvancedTab::render($options);
				return;
			case 'basic':
			default:
				BasicTab::render($options);
				return;
		}
	}

	private static function render_notice(string $notice): void
	{
		if ($notice === '') {
			return;
		}

		if ($notice === 'cr_update_checked') {
			$status = isset($_GET['cr_update_status']) ? sanitize_key((string)$_GET['cr_update_status']) : '';
			$remote = isset($_GET['cr_update_remote']) ? sanitize_text_field((string)$_GET['cr_update_remote']) : '';
			$checked_at = isset($_GET['cr_update_checked_at']) ? (int)$_GET['cr_update_checked_at'] : 0;
			$current = defined('CR_VERSION') ? (string)CR_VERSION : '';

			$cls = 'notice notice-info';
			$msg = '已检查更新。';
			if ($status === 'available' && $remote !== '' && $current !== '') {
				$cls = 'notice notice-warning';
				$msg = '已检查更新：发现新版本 ' . $remote . '（当前 ' . $current . '）。你可在「插件」页一键更新，或打开更新日志查看说明与下载链接。';
			} elseif ($status === 'latest' && $current !== '') {
				$msg = '已检查更新：当前已是最新版本（' . $current . '）。';
			} elseif ($status === 'failed') {
				$cls = 'notice notice-error';
				$msg = '检查更新失败：未能获取最新版本信息，请稍后重试。';
			}

			if ($checked_at > 0) {
				$msg .= '（检查时间：' . wp_date('Y-m-d H:i', $checked_at) . '）';
			}

			echo '<div class="' . esc_attr($cls) . '"><p>' . esc_html($msg) . '</p></div>';
			return;
		}

		if ($notice === 'cr_jsonld_probe_done' || $notice === 'cr_jsonld_probe_failed') {
			$res = get_transient('cr_jsonld_probe_result_v1');
			$res = is_array($res) ? $res : [];
			$at = (int)($res['at'] ?? 0);
			$total = (int)($res['total'] ?? 0);
			$ok = (int)($res['ok'] ?? 0);
			$skipped = (int)($res['skipped'] ?? 0);
			$failed = (int)($res['failed'] ?? 0);

			$cls = 'notice notice-info';
			if ($failed > 0 && $ok > 0) {
				$cls = 'notice notice-warning';
			} elseif ($failed > 0 && $ok <= 0) {
				$cls = 'notice notice-error';
			}

			$msg = '已完成 JSON-LD 一键探测。';
			if ($total > 0) {
				$msg .= '成功 ' . $ok . '，跳过 ' . $skipped . '，失败 ' . $failed . '（共 ' . $total . '）。';
			}
			if ($at > 0) {
				$msg .= '（时间：' . wp_date('Y-m-d H:i:s', $at) . '）';
			}
			if ($failed > 0) {
				$msg .= '失败通常是站点无法完成 loopback 请求（被防火墙/安全插件拦截或 DNS/SSL 问题）。';
			}
			if ($skipped > 0) {
				$msg .= '跳过通常是站点暂无对应页面样本（例如没有已发布页面/分类）。';
			}

			echo '<div class="' . esc_attr($cls) . '"><p>' . esc_html($msg) . '</p></div>';
			return;
		}

		$messages = [
			'cr_saved' => '设置已保存。',
			'cr_license_verified' => '许可证校验成功：已解锁 Pro。',
			'cr_license_verify_failed' => '许可证校验未通过：请检查 key 与域名绑定，或稍后重试。',
			'cr_import_success' => '导入成功，配置已生效。',
			'cr_import_missing_file' => '导入失败：未选择文件。',
			'cr_import_invalid_json' => '导入失败：JSON 解析失败。',
			'cr_import_version_mismatch' => '导入失败：配置版本不匹配（可能来自旧版本导出文件）。',
		];
		$msg = $messages[$notice] ?? $notice;
		echo '<div class="notice notice-info"><p>' . esc_html($msg) . '</p></div>';
	}

	private static function render_quickstart(array $options, string $active_tab): void
	{
		// 仅在基础设置页展示，避免对其它 Tab 造成干扰。
		if ($active_tab !== 'basic') {
			return;
		}

		$ai_url = add_query_arg(['page' => 'content-ready-settings', 'tab' => 'ai_privacy'], admin_url('admin.php'));
		$modules_url = add_query_arg(['page' => 'content-ready-settings', 'tab' => 'modules_output'], admin_url('admin.php'));
		$assessment_url = admin_url('admin.php?page=content-ready-assessment');
		$posts_url = admin_url('edit.php');

		$ai = is_array($options['ai'] ?? null) ? $options['ai'] : [];
		$ai_configured = trim((string)($ai['base_url'] ?? '')) !== '' && trim((string)($ai['api_key'] ?? '')) !== '' && trim((string)($ai['model'] ?? '')) !== '';

		echo '<div class="cr-settings-cards">';

		echo '<div class="cr-settings-card">';
		echo '<h2>快速开始（3 步）</h2>';
		echo '<ul>';
		echo '<li>第 1 步：配置 AI（必需）' . ($ai_configured ? '：已配置 ✅' : '：未配置 ⚠️') . '</li>';
		echo '<li>第 2 步：先用默认配置试跑 1 篇文章（推荐先手动生成）</li>';
		echo '<li>第 3 步：按需再调整模块/结构化/自动化（可选）</li>';
		echo '</ul>';
		echo '<div class="cr-settings-card__actions">';
		echo '<a class="button button-primary" href="' . esc_url($ai_url) . '">去配置 AI</a>';
		echo '<a class="button" href="' . esc_url($assessment_url) . '">开始评估</a>';
		echo '<a class="button" href="' . esc_url($posts_url) . '">管理文章</a>';
		echo '<a class="button" href="' . esc_url($modules_url) . '">（可选）调整模块</a>';
		echo '</div>';
		echo '<p class="description">提示：只要先把 AI 配好，就可以开始使用；其它功能可以先保持默认，后续再慢慢优化。</p>';
		echo '</div>';

		echo '</div>';
	}



}
