<?php

declare(strict_types=1);

namespace ContentReady\Admin\Settings;

use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Dispatch as ModulesDispatch;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class Options
{
	public static function get_options(string $option_key): array
	{
		$options = get_option($option_key, []);
		$options = is_array($options) ? $options : [];
		$options = Capabilities::normalize_options($options);
		$options['page_types'] = PageTypesConfig::normalize_page_types($options['page_types'] ?? null);

		// 固定模块顺序与位置（不让站长用户配置排序/位置，避免心智负担）。
		$options['modules_order'] = ModulesConfig::default_order();
		$options['modules'] = ModulesConfig::normalize_modules_config($options['modules'] ?? null);

		// 轻量迁移：旧版本的“verdict”标签为「一句话结论」，新版调整为更准确的「核心结论」。
		// 仅在用户未自定义标签时自动替换（label 恰好等于旧默认值）。
		if (is_array($options['modules']['verdict'] ?? null)) {
			$verdict_label = (string)($options['modules']['verdict']['label'] ?? '');
			if ($verdict_label === '一句话结论') {
				$options['modules']['verdict']['label'] = '核心结论';
			}
		}

		// 模块调度策略（Pro-only）：用于“启用很多模块”与“轻量精选推荐组合”两类用户。
		$is_pro = Capabilities::is_pro($options);
		$gen = is_array($options['generation'] ?? null) ? $options['generation'] : [];
		$raw_mode = isset($gen['dispatch_mode']) ? (string)$gen['dispatch_mode'] : '';
		$mode = ModulesDispatch::normalize_mode($raw_mode, $is_pro);

		// 兼容旧版本：只有 adaptive_enabled 时，自动映射为 adaptive。
		if ($raw_mode === '' && !array_key_exists('dispatch_mode', $gen) && !empty($gen['adaptive_enabled'])) {
			$mode = ModulesDispatch::MODE_ADAPTIVE;
		}

		if (!is_array($options['generation'] ?? null)) {
			$options['generation'] = [];
		}
		$options['generation']['dispatch_mode'] = $mode;
		$options['generation']['adaptive_enabled'] = ($mode === ModulesDispatch::MODE_ADAPTIVE);

		if ($mode === ModulesDispatch::MODE_LIGHTWEIGHT) {
			$options['modules'] = ModulesDispatch::apply_lightweight_modules_config($options['modules']);
		}

		return StructuredOutputNormalizer::normalize($options);
	}
}

