<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Admin\Settings\Assets;
use ContentReady\Admin\Settings\Defaults;
use ContentReady\Admin\Settings\Menu;
use ContentReady\Admin\Settings\Options;
use ContentReady\Admin\Settings\PageEntry;

final class Settings
{
	public const OPTION_KEY = 'cr_options_v1';
	public const CONFIG_VERSION = 1;
	/** @var array<string,array> */
	private static array $options_cache = [];

	public static function register(): void
	{
		add_action('admin_menu', [self::class, 'add_menu']);
		add_action('admin_enqueue_scripts', [self::class, 'enqueue_assets']);
		add_action('admin_post_cr_save_settings', [self::class, 'handle_save']);
		add_action('update_option_' . self::OPTION_KEY, [self::class, 'on_options_updated'], 10, 2);
		ImportExport::register();
		JsonLdProbe::register();
		AiProbe::register();
	}

	public static function enqueue_assets(string $hook): void
	{
		Assets::enqueue_assets($hook);
	}

	public static function maybe_seed_defaults(): void
	{
		Defaults::maybe_seed_defaults(self::OPTION_KEY, self::default_options());
	}

	public static function get_options(bool $force_refresh = false): array
	{
		if (
			!$force_refresh &&
			array_key_exists(self::OPTION_KEY, self::$options_cache) &&
			is_array(self::$options_cache[self::OPTION_KEY])
		) {
			return self::$options_cache[self::OPTION_KEY];
		}

		$options = Options::get_options(self::OPTION_KEY);
		self::$options_cache[self::OPTION_KEY] = $options;
		return $options;
	}

	public static function get_raw_options(): array
	{
		$options = get_option(self::OPTION_KEY, []);
		return is_array($options) ? $options : [];
	}

	public static function update_options(array $options): void
	{
		update_option(self::OPTION_KEY, $options, false);
		self::reset_options_cache();
	}

	public static function default_options(): array
	{
		return Defaults::build(self::CONFIG_VERSION);
	}

	public static function add_menu(): void
	{
		Menu::register();
	}

	public static function render_page(): void
	{
		PageEntry::render();
	}

	public static function handle_save(): void
	{
		SettingsSaver::handle_save();
	}

	public static function on_options_updated($old_value, $value): void
	{
		self::reset_options_cache();
	}

	public static function reset_options_cache(): void
	{
		unset(self::$options_cache[self::OPTION_KEY]);
	}
}
