<?php

declare(strict_types=1);

namespace ContentReady\Admin\JsonLdProbe;

final class TargetsBuilder
{
	public static function build_targets(): array
	{
		$targets = [
			'home' => [
				'label' => '首页',
				'url' => home_url('/'),
			],
			'search' => [
				'label' => '搜索结果页',
				'url' => get_search_link('content-ready'),
			],
		];

		$term_url = self::pick_term_url();
		if ($term_url !== '') {
			$targets['taxonomy'] = [
				'label' => '分类/标签页',
				'url' => $term_url,
			];
		} else {
			$targets['taxonomy'] = [
				'label' => '分类/标签页',
				'url' => '',
			];
		}

		$post_id = self::pick_post_id('post');
		$targets['singular_post'] = [
			'label' => '文章页（post）',
			'post_id' => $post_id,
			'url' => $post_id > 0 ? (string)get_permalink($post_id) : '',
		];

		$page_id = self::pick_post_id('page');
		$targets['singular_page'] = [
			'label' => '页面（page）',
			'post_id' => $page_id,
			'url' => $page_id > 0 ? (string)get_permalink($page_id) : '',
		];

		$cpt = self::pick_custom_post();
		$targets['singular_cpt'] = [
			'label' => '自定义内容（CPT）',
			'post_id' => (int)($cpt['post_id'] ?? 0),
			'url' => (string)($cpt['url'] ?? ''),
		];

		return $targets;
	}

	private static function pick_post_id(string $post_type): int
	{
		$ids = get_posts([
			'post_type' => $post_type,
			'post_status' => 'publish',
			'posts_per_page' => 1,
			'orderby' => 'date',
			'order' => 'DESC',
			'fields' => 'ids',
			'no_found_rows' => true,
		]);
		$ids = is_array($ids) ? $ids : [];
		return $ids !== [] ? (int)$ids[0] : 0;
	}

	private static function pick_custom_post(): array
	{
		$types = get_post_types(['public' => true, '_builtin' => false], 'objects');
		$types = is_array($types) ? $types : [];
		foreach ($types as $obj) {
			if (!is_object($obj)) {
				continue;
			}
			$name = isset($obj->name) ? (string)$obj->name : '';
			if ($name === '') {
				continue;
			}
			$publicly_queryable = isset($obj->publicly_queryable) ? (bool)$obj->publicly_queryable : true;
			if (!$publicly_queryable) {
				continue;
			}

			$id = self::pick_post_id($name);
			if ($id <= 0) {
				continue;
			}

			$url = (string)get_permalink($id);
			if ($url === '') {
				continue;
			}

			return [
				'post_type' => $name,
				'post_id' => $id,
				'url' => $url,
			];
		}

		return [];
	}

	private static function pick_term_url(): string
	{
		$taxes = ['category', 'post_tag'];
		$terms = get_terms([
			'taxonomy' => $taxes,
			'hide_empty' => true,
			'number' => 1,
		]);
		$terms = is_array($terms) ? $terms : [];
		if ($terms === []) {
			return '';
		}

		$link = get_term_link($terms[0]);
		if (is_wp_error($link)) {
			return '';
		}

		return esc_url_raw((string)$link);
	}
}

