<?php

declare(strict_types=1);

namespace ContentReady\Admin\JsonLdProbe;

use ContentReady\Meta\MetaStore;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class ScopeStatusEvaluator
{
	public static function evaluate_scope_status(
		array $options,
		string $scope,
		string $mode,
		int $post_id,
		int $other_scripts,
		bool &$would_output,
		string &$blocked_reason
	): void {
		$would_output = false;
		$blocked_reason = '';

		$so = is_array($options['structured_output'] ?? null) ? $options['structured_output'] : [];
		$enabled = !empty($so['enabled']);
		if (!$enabled || $mode === 'off') {
			$blocked_reason = 'structured_disabled';
			return;
		}

		if (!self::has_objects_for_scope($options, $scope, $post_id)) {
			$blocked_reason = 'no_objects_enabled';
			return;
		}

		if ($mode === 'auto' && $other_scripts > 0) {
			$blocked_reason = 'auto_conflict_detected';
			return;
		}

		if ($mode === 'auto' && $post_id > 0 && in_array($scope, ['singular_post', 'singular_page', 'singular_cpt'], true)) {
			$meta = MetaStore::get($post_id);
			$compat = is_array($meta['compat'] ?? null) ? $meta['compat'] : [];
			$checked_at = (int)($compat['checked_at'] ?? 0);
			$conflict = !empty($compat['jsonld_conflict']);
			$last_error = sanitize_text_field((string)($compat['last_error'] ?? ''));

			if ($checked_at <= 0) {
				$blocked_reason = 'auto_not_checked_yet';
				return;
			}
			if ($last_error !== '') {
				$blocked_reason = 'auto_check_failed';
				return;
			}
			if ($conflict) {
				$blocked_reason = 'auto_conflict_detected';
				return;
			}
		}

		$would_output = true;
	}

	private static function has_objects_for_scope(array $options, string $scope, int $post_id): bool
	{
		$so = is_array($options['structured_output'] ?? null) ? $options['structured_output'] : [];
		$objects = is_array($so['objects'] ?? null) ? $so['objects'] : [];

		$want_site_graph = !empty($objects['site_graph']);
		$want_webpage = !empty($objects['webpage']);
		if ($want_site_graph || $want_webpage) {
			return true;
		}

		if (!in_array($scope, ['singular_post', 'singular_page', 'singular_cpt'], true)) {
			return false;
		}
		if ($post_id <= 0) {
			return false;
		}

		$meta = MetaStore::get($post_id);
		$page_type = sanitize_key((string)($meta['page_type'] ?? 'article'));
		$strategy = PageTypesConfig::schema_strategy_for($page_type, $options);
		$schema_profile = sanitize_key((string)($strategy['profile'] ?? 'article'));
		if (!in_array($schema_profile, ['article', 'review', 'product', 'service', 'howto'], true)) {
			return false;
		}

		if (!empty($objects['article'])) {
			return true;
		}

		if (!empty($objects['faq'])) {
			$faq_items = $meta['modules']['faq']['data']['items'] ?? null;
			if (is_array($faq_items) && $faq_items !== []) {
				return true;
			}
		}

		return false;
	}
}

