<?php

declare(strict_types=1);

namespace ContentReady\Admin\JsonLdProbe;

use ContentReady\Compat\JsonLdDetector;

final class Runner
{
	public static function run(array $options, string $status_transient_key): array
	{
		$targets = TargetsBuilder::build_targets();

		$result = [
			'at' => time(),
			'total' => count($targets),
			'ok' => 0,
			'skipped' => 0,
			'failed' => 0,
			'details' => [],
		];

		foreach ($targets as $scope_id => $t) {
			$scope = sanitize_key((string)$scope_id);
			$label = sanitize_text_field((string)($t['label'] ?? $scope));
			$url = esc_url_raw((string)($t['url'] ?? ''));
			$post_id = (int)($t['post_id'] ?? 0);

			if ($scope === '' || $url === '') {
				$result['skipped']++;
				$result['details'][$scope] = [
					'label' => $label,
					'status' => 'skipped',
					'reason' => 'missing_sample',
				];
				continue;
			}

			$mode = ModeResolver::effective_mode($options, $scope, $post_id);
			if ($post_id > 0 && $mode === 'auto') {
				JsonLdDetector::run_check($post_id);
			}

			$resp = wp_remote_get($url, [
				'timeout' => 10,
				'redirection' => 3,
				'user-agent' => 'ContentReady/' . (defined('CR_VERSION') ? (string)CR_VERSION : '0') . ' Probe',
			]);

			if (is_wp_error($resp)) {
				$result['failed']++;
				$result['details'][$scope] = [
					'label' => $label,
					'status' => 'failed',
					'url' => $url,
					'error' => sanitize_text_field($resp->get_error_message()),
				];
				continue;
			}

			$code = (int)wp_remote_retrieve_response_code($resp);
			if ($code >= 400) {
				$result['failed']++;
				$result['details'][$scope] = [
					'label' => $label,
					'status' => 'failed',
					'url' => $url,
					'http_code' => $code,
				];
				continue;
			}

			$body = (string)wp_remote_retrieve_body($resp);
			$head = Html::extract_head($body);
			$other_scripts = Html::count_other_jsonld_scripts($head);

			$would_output = false;
			$blocked_reason = '';
			ScopeStatusEvaluator::evaluate_scope_status($options, $scope, $mode, $post_id, $other_scripts, $would_output, $blocked_reason);
			StatusStore::write_scope_status($status_transient_key, $scope, $mode, $would_output, $blocked_reason, $other_scripts);

			$result['ok']++;
			$result['details'][$scope] = [
				'label' => $label,
				'status' => 'ok',
				'url' => $url,
				'http_code' => $code,
				'other_jsonld' => $other_scripts,
			];
		}

		return $result;
	}
}

