<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Admin\JsonLdProbe\Runner;

final class JsonLdProbe
{
	private const RESULT_TRANSIENT_KEY = 'cr_jsonld_probe_result_v1';
	private const STATUS_TRANSIENT_KEY = 'cr_jsonld_scope_status_v1';

	public static function register(): void
	{
		add_action('admin_post_cr_probe_jsonld', [self::class, 'handle']);
		add_action('wp_ajax_cr_probe_jsonld', [self::class, 'handle_ajax']);
	}

	public static function handle(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have permission to access this page.', 'content-ready'));
		}

		check_admin_referer('cr_probe_jsonld');

		$default_redirect = add_query_arg(
			['page' => 'content-ready-settings', 'tab' => 'modules_output'],
			admin_url('admin.php')
		);

		$redirect = isset($_GET['redirect']) ? esc_url_raw((string)$_GET['redirect']) : '';
		$redirect = $redirect !== '' ? wp_validate_redirect($redirect, $default_redirect) : $default_redirect;

		$options = Settings::get_options();
		$result = Runner::run($options, self::STATUS_TRANSIENT_KEY);

		set_transient(self::RESULT_TRANSIENT_KEY, $result, 10 * MINUTE_IN_SECONDS);

		$notice = $result['failed'] > 0 ? 'cr_jsonld_probe_failed' : 'cr_jsonld_probe_done';
		wp_safe_redirect(add_query_arg('cr_notice', $notice, $redirect));
		exit;
	}

	public static function handle_ajax(): void
	{
		if (!current_user_can('manage_options')) {
			wp_send_json_error(['message' => 'permission_denied'], 403);
		}

		check_ajax_referer('cr_probe_jsonld');

		$options = Settings::get_options();
		$result = Runner::run($options, self::STATUS_TRANSIENT_KEY);
		set_transient(self::RESULT_TRANSIENT_KEY, $result, 10 * MINUTE_IN_SECONDS);

		$status = get_transient(self::STATUS_TRANSIENT_KEY);
		$status = is_array($status) ? $status : [];

		wp_send_json_success([
			'result' => $result,
			'status' => $status,
		]);
	}
}
