<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class ImportExport
{
	public static function register(): void
	{
		add_action('admin_post_cr_export_config', [self::class, 'handle_export']);
		add_action('admin_post_cr_import_config', [self::class, 'handle_import']);
	}

	public static function handle_export(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die('Forbidden');
		}
		check_admin_referer('cr_export_config');

		$type = isset($_GET['type']) ? sanitize_text_field((string)$_GET['type']) : 'full';
		$options = Settings::get_options();

		$payload = [
			'config_version' => Settings::CONFIG_VERSION,
			'exported_at' => time(),
			'type' => $type,
		];

		if ($type === 'site' || $type === 'full') {
			$payload['site'] = self::normalize_site($options['site'] ?? null);
			$payload['ui'] = self::normalize_ui($options['ui'] ?? null);
		}

		if ($type === 'modules' || $type === 'full') {
			$payload['modules'] = ModulesConfig::sanitize_modules_config($options['modules'] ?? null);
		}
		if ($type === 'page_types' || $type === 'full') {
			$payload['page_types'] = self::normalize_page_types($options['page_types'] ?? null);
		}

		$filename = 'content-ready-config-' . $type . '-v' . Settings::CONFIG_VERSION . '.json';
		$json = wp_json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
		if (!is_string($json)) {
			wp_die('Export failed');
		}

		nocache_headers();
		header('Content-Type: application/json; charset=utf-8');
		header('Content-Disposition: attachment; filename="' . $filename . '"');
		echo $json;
		exit;
	}

	public static function handle_import(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die('Forbidden');
		}
		check_admin_referer('cr_import_config');

		$type = isset($_POST['type']) ? sanitize_text_field((string)$_POST['type']) : 'full';
		$file = $_FILES['config_file'] ?? null; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput

		if (!is_array($file) || empty($file['tmp_name'])) {
			self::redirect_with_notice('cr_import_missing_file');
		}

		$raw = file_get_contents((string)$file['tmp_name']);
		$decoded = is_string($raw) ? json_decode($raw, true) : null;
		if (!is_array($decoded)) {
			self::redirect_with_notice('cr_import_invalid_json');
		}

		$version = (int)($decoded['config_version'] ?? 0);
		if ($version !== Settings::CONFIG_VERSION) {
			self::redirect_with_notice('cr_import_version_mismatch');
		}

		$options = Settings::get_options();

		if ($type === 'site' || $type === 'full') {
			$options['site'] = self::normalize_site($decoded['site'] ?? null);
			$options['ui'] = self::normalize_ui($decoded['ui'] ?? null);
		}

		if ($type === 'modules' || $type === 'full') {
			$options['modules_order'] = ModulesConfig::default_order();
			$options['modules'] = ModulesConfig::sanitize_modules_config($decoded['modules'] ?? null);
		}

		if ($type === 'page_types' || $type === 'full') {
			$options['page_types'] = self::normalize_page_types($decoded['page_types'] ?? null);
		}

		$options['config_version'] = Settings::CONFIG_VERSION;
		Settings::update_options($options);
		self::redirect_with_notice('cr_import_success');
	}

	private static function normalize_page_types($value): array
	{
		return PageTypesConfig::normalize_page_types($value);
	}

	private static function normalize_site($value): array
	{
		$value = is_array($value) ? $value : [];
		$out = [
			'display_name' => sanitize_text_field((string)($value['display_name'] ?? '')),
			'logo_url' => esc_url_raw((string)($value['logo_url'] ?? '')),
			'locale' => sanitize_text_field((string)($value['locale'] ?? '')),
			'date_format' => sanitize_text_field((string)($value['date_format'] ?? 'wp')),
			'author_mode' => sanitize_key((string)($value['author_mode'] ?? 'wp')),
			'brand_author_name' => sanitize_text_field((string)($value['brand_author_name'] ?? '')),
			'entity_type' => sanitize_key((string)($value['entity_type'] ?? 'organization')),
			'entity_name' => sanitize_text_field((string)($value['entity_name'] ?? '')),
			'entity_url' => esc_url_raw((string)($value['entity_url'] ?? '')),
			'entity_same_as' => sanitize_textarea_field((string)($value['entity_same_as'] ?? '')),
			'entity_phone' => sanitize_text_field((string)($value['entity_phone'] ?? '')),
			'entity_address' => sanitize_text_field((string)($value['entity_address'] ?? '')),
			'entity_description' => sanitize_textarea_field((string)($value['entity_description'] ?? '')),
		];
		if (!in_array($out['author_mode'], ['wp', 'brand'], true)) {
			$out['author_mode'] = 'wp';
		}
		if (!in_array($out['entity_type'], ['organization', 'person', 'local_business'], true)) {
			$out['entity_type'] = 'organization';
		}
		if ($out['date_format'] === '') {
			$out['date_format'] = 'wp';
		}
		return $out;
	}

	private static function normalize_ui($value): array
	{
		$value = is_array($value) ? $value : [];
		return [
			'editor_entry_label' => sanitize_text_field((string)($value['editor_entry_label'] ?? '')),
		];
	}

	private static function redirect_with_notice(string $code): void
	{
		$url = add_query_arg(
			['page' => 'content-ready', 'cr_notice' => $code],
			admin_url('admin.php')
		);
		wp_safe_redirect($url);
		exit;
	}
}
